<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Team extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'runok_team';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Team', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
		
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                    'layout-2' => esc_html__('Layout 2', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'background_shape',
            [
                'label' => esc_html__( 'Background Shape', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__('Subtitle', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('Team Members', 'delishs-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Meet The Expert Chef', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title_shape',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_team_list',
            [
                'label' => esc_html__('Members', 'delishs-core'),
            ]
        );

        $repeater = new Repeater();                  

        $repeater->add_control(
            'image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __( 'Image', 'delishs-core' ),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );                      

        $repeater->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => __( 'Name', 'delishs-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'designation',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'show_label' => true,
                'label' => __( 'Designation', 'delishs-core' ),
                'default' => __( 'Software Tester', 'delishs-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'delishs_link_switcher',
            [
                'label' => esc_html__( 'Show Link', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'delishs-core' ),
                'label_off' => esc_html__( 'No', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'delishs_link_type',
            [
                'label' => esc_html__( 'Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'delishs_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'delishs_link',
            [
                'label' => esc_html__( 'Link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'delishs_link_type' => '1',
                    'delishs_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'delishs_page_link',
            [
                'label' => esc_html__( 'Select Link Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'delishs_link_type' => '2',
                    'delishs_link_switcher' => 'yes',
                ]
            ]
        );

        // REPEATER
        $this->add_control(
            'teams',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '<# print(title || "Carousel Item"); #>',
                'default' => [
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'title' => __( 'KUMAN TUNMAN', 'delishs-core' ),
                        'designation' => __( 'WAITER', 'delishs-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'title' => __( 'MATTHEW GEORGE', 'delishs-core' ),
                        'designation' => __( 'MANAGER', 'delishs-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'title' => __( 'CHARLES DANIEL', 'delishs-core' ),
                        'designation' => __( 'HEAD CHEF', 'delishs-core' ),
                    ],
                ]
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background_two',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-bottom__background' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => __( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        // Title
        $this->add_control(
            '_heading_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Subheading', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'subheading_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'subheading_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'subheading_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'subheading_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'subheading_border_radius',
            [
                'label' => __( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Title
        $this->add_control(
            '_section_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_title_layout',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Layout', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_layout_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section__title-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'team_member_style',
			[
				'label' => __( 'Members', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        // Icon
        $this->add_control(
            '_heading_icon_wrapper',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon Wrapper', 'delishs-core' ),
            ]
        );

        $this->start_controls_tabs( 'tabs_icon_wrapper_style' );

        $this->start_controls_tab(
            'icon_wrapper_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'icon_wrapper_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .expand' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-social .expand' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_wrapper_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .expand' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-social .expand' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_wrapper_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'icon_wrapper_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .expand:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-box:hover .team-social .expand' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_wrapper_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .expand:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box:hover .team-social .expand' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        // Icon
        $this->add_control(
            '_heading_icon',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'delishs-core' ),
            ]
        );

        $this->start_controls_tabs( 'tabs_info_icon_type' );

        $this->start_controls_tab(
            'icon_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .social-list li a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-social .social-list li a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .social-list li a' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-social .social-list li a' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .social-list li a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-box .team-social .social-list li a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-thumb .team-social .social-list li a:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-social .social-list li a:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        // Name
        $this->add_control(
            '_heading_member_name',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Name', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'member_name_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .team-box .team-content .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
            'member_name_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content .title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-content .title' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'member_name_color_hover',
            [
                'label' => __( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content .title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-content .title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box:hover .team-content .title a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'member_name_typography',
                'selector' => '{{WRAPPER}} .team-item .team-content .title, .team-box .team-content .title',
            ]
        );

        // Name
        $this->add_control(
            '_heading_member_designation',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Designation', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

		$this->add_control(
            'member_designation_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-content span' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'member_designation_color_hover',
            [
                'label' => __( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-box:hover .team-content span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'member_designation_typography',
                'selector' => '{{WRAPPER}} .team-item .team-content span, .team-box .team-content span',
            ]
        );

        $this->add_control(
            '_heading_name_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

		$this->add_control(
            'member_name_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box:hover .team-content' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'member_name_background_two',
            [
                'label' => __( 'Background Shape', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-box .team-content:after' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-box .team-content:before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'member_name_border',
				'selector' => '{{WRAPPER}} .team-box .team-content',
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
			]
		);

        $this->add_responsive_control(
            'member_name_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .team-item .team-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-box:hover .team-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>

                <section class="delishs-el-section team team-bottom__up section-space">
                    <div class="container">
                        <div class="row g-24">
                            <div class="col-12">
                                <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                            <?php echo rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title'])): ?>
                                        <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                            <?php echo rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title_shape'])): ?>
                                        <div class="dot-shpae wow clip-a-z">
                                            <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24 mb-minus-24">
                            <?php foreach ( $settings['teams'] as $item ) :
                                $name = delishs_kses( $item['title' ] );

                                if ( !empty($item['image']['url']) ) {
                                    $delishs_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $delishs_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                
                                // Link
                                if ('2' == $item['delishs_link_type']) {
                                    $link = get_permalink($item['delishs_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['delishs_link']['url']) ? $item['delishs_link']['url'] : '';
                                    $target = !empty($item['delishs_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['delishs_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="col-lg-4 col-md-6">
                                <div class="team-item mb-24">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="team-item__media">
                                        <img src="<?php echo esc_url($delishs_team_image_url); ?>" class="img-fluid" alt="<?php echo esc_attr($delishs_team_image_alt); ?>">
                                    </a>
                                    <div class="team-item__text text-center">
                                        <h5 class="text-uppercase">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo delishs_kses( $item['title'] ); ?>
                                            </a>
                                        </h5>
                                        <?php if( !empty($item['designation']) ) : ?>
                                            <span><?php echo delishs_kses( $item['designation'] ); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>
                <?php if ( !empty ($settings['background_shape']) ) : ?>
                    <div class="team-bottom__background" data-background="<?php print get_template_directory_uri(); ?>/assets/imgs/team/team-bottom__background.png"></div>
                <?php endif; ?>

            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <section class="delishs-el-section team-2 section-space__bottom-100">
                    <div class="container">
                        <div class="row g-24">
                            <div class="col-12">
                                <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                            <?php echo rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title'])): ?>
                                        <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                            <?php echo rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title_shape'])): ?>
                                        <div class="dot-shpae wow clip-a-z">
                                            <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24 mb-minus-24">
                            <?php foreach ( $settings['teams'] as $item ) :
                                $name = delishs_kses( $item['title' ] );

                                if ( !empty($item['image']['url']) ) {
                                    $delishs_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $delishs_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                
                                // Link
                                if ('2' == $item['delishs_link_type']) {
                                    $link = get_permalink($item['delishs_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['delishs_link']['url']) ? $item['delishs_link']['url'] : '';
                                    $target = !empty($item['delishs_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['delishs_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="col-xl-3 col-lg-4 col-sm-6">
                                <div class="team-2__item mb-24">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="team-2__item-media">
                                        <img class="img-fluid" src="<?php echo esc_url($delishs_team_image_url); ?>" alt="<?php echo esc_attr($delishs_team_image_alt); ?>">
                                    </a>

                                    <div class="team-2__item-text">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="team-2__item-arrow">
                                            <svg width="8" height="12" viewBox="0 0 8 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M5.59792 5.86858L0 0.737154L0.675725 0L7.07781 5.86858L0.675725 11.7372L0 11L5.59792 5.86858Z" fill="white"/>
                                            </svg>
                                        </a>
                                        <?php if( !empty($item['designation']) ) : ?>
                                            <span><?php echo delishs_kses( $item['designation'] ); ?></span>
                                        <?php endif; ?>
                                        <h6 class="text-uppercase">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo delishs_kses( $item['title'] ); ?>
                                            </a>
                                        </h6>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Delishs_Team() );