<?php

namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delishs Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Services extends Widget_Base {

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'delishs_services';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return __( 'Services', 'delishs-core' );
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'delishs-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return [ 'delishs_core' ];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends() {
        return [ 'delishs-services' ];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
            ]
        );
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                    'layout-2' => esc_html__('Layout 2', 'delishs-core'),
                    'layout-3' => esc_html__('Layout 3', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Our Services', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Our Restaurant Service', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title_shape',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_service',
            [
                'label' => esc_html__('Services List', 'delishs-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'service_image',
            [
                'label' => esc_html__('Upload Image', 'delishs-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

		$repeater->add_control(
			'service_title',
			[
				'label' => esc_html__( 'Title', 'delishs-core' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'service_description',
			[
				'label' => esc_html__( 'Description', 'delishs-core' ),
				'type' => Controls_Manager::TEXTAREA,
				'label_block' => true,
			]
		);

        $repeater->add_control(
			'_heading_content_service_button',
			[
				'label' => esc_html__( 'Button', 'delishs-core' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $repeater->add_control(
            'service_button_text',
            [
                'label' => esc_html__('Button Text', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'delishs-core'),
                'title' => esc_html__('Enter button text', 'delishs-core'),
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );
        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                ]
            ]
        );
        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
			'service_list',
			[
				'label' => esc_html__( 'Repeater List', 'delishs-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'service_title' => esc_html__( 'Wedding Service', 'delishs-core' ),
						'service_description' => esc_html__( 'The chocolate fondant which did not disappoint It was rich and indulgent, with a gooey center that was pure bliss.', 'delishs-core' ),
					],
					[
						'service_title' => esc_html__( 'Buffet Service', 'delishs-core' ),
						'service_description' => esc_html__( 'Assertively myocardinate robust e-tailers for extensible human capital. dpropriately benchmark networks.', 'delishs-core' ),
					],
					[
						'service_title' => esc_html__( 'Party Service', 'delishs-core' ),
						'service_description' => esc_html__( 'Assertively myocardinate robust e-tailers for extensible human capital. dpropriately benchmark networks.', 'delishs-core' ),
					],
				],
				'title_field' => '{{{ service_title }}}',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_service_two',
            [
                'label' => esc_html__('Services List', 'delishs-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'service_image',
            [
                'label' => esc_html__('Upload Image', 'delishs-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
			'_heading_content_service_button',
			[
				'label' => esc_html__( 'Button', 'delishs-core' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $repeater->add_control(
            'service_button_text',
            [
                'label' => esc_html__('Button Text', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'delishs-core'),
                'title' => esc_html__('Enter button text', 'delishs-core'),
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );
        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                ]
            ]
        );
        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
			'service_list_two',
			[
				'label' => esc_html__( 'Repeater List', 'delishs-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'service_button_text' => esc_html__( 'Tea & Drinks', 'delishs-core' ),
					],
					[
						'service_button_text' => esc_html__( 'Lunch Meal', 'delishs-core' ),
					],
					[
						'service_button_text' => esc_html__( 'Dinner Meal', 'delishs-core' ),
					],
					[
						'service_button_text' => esc_html__( 'Desserts Menu', 'delishs-core' ),
					],
				],
				'title_field' => '{{{ service_button_text }}}',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => esc_html__( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_layout_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .section__title-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // TAB_STYLE
        $this->start_controls_section(
            '_style_service_list',
            [
                'label' => esc_html__( 'Services List', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_service_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'service_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .our-service__item-content h5' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-2__item-text h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'service_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service__item-content h5 a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text h6' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'service_title_color_hover',
            [
                'label' => esc_html__( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service__item-content h5 a:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_title_typography',
                'selector' => '{{WRAPPER}} .our-service__item-content h5, .service-2__item-text h6',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_service_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'service_description_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .our-service__item-content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-2__item-text p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'service_description_color',
            [
                'label' => esc_html__( 'Text', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service__item-content p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_description_typography',
                'selector' => '{{WRAPPER}} .our-service__item-content p, .service-2__item-text p',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_service_link',
            [
                'label' => esc_html__( 'Link', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'service_link_spacing',
            [
                'label' => esc_html__( 'Top Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .rr-btn' => 'margin-top: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-2__item-text .btn-book' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->start_controls_tabs( 'service_link_tabs' );
        
        $this->start_controls_tab(
            'service_link_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'service_link_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-one' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item .btn' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text .btn-book' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'service_link_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item .btn' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item-media::before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item-media::after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item-media::after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text .btn-book' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'service_link_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'service_link_color_hover',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-two' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item:hover .btn' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text .btn-book:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'service_link_border_color_hover',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn:after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item:hover .btn' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item:hover .our-service-2__item-media::before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .our-service-2__item:hover .our-service-2__item-media::after' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item-text .btn-book:hover' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_link_typography',
                'selector' => '{{WRAPPER}} .rr-btn, .our-service-2__item .btn, .service-2__item-text .btn-book',
            ],
        );

        $this->add_control(
            '_heading_style_service_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'service_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-service__item' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .service-2__item' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        ?>

        <?php if ( $settings['design_style']  == 'layout-1' ): ?>

            <section class="delishs-el-section our-service section-space section-bg-1" data-background="assets/imgs/service/service-bg.png">
                <div class="container">
                    <div class="row g-24">
                        <div class="col-12">
                            <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                    <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                        <?php print rrdevs_kses($settings['section_subheading']); ?>
                                    </span>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                    <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                        <?php print rrdevs_kses($settings['section_title']); ?>
                                    </h2>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                    <div class="dot-shpae wow clip-a-z">
                                        <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="row g-24 mb-minus-24">
                        <?php foreach ($settings['service_list'] as $index => $item) : 
                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }
                            
                            ?>
                            <div class="col-xl-4 col-md-6">
                                <div class="our-service__item white-bg mb-24">
                                    <?php if( !empty ($service_image) ): ?>
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="our-service__item-media d-block mb-30">
                                            <img src="<?php print esc_url($service_image); ?>" alt="<?php print esc_attr($service_image_alt); ?>">
                                        </a>
                                    <?php endif; ?>

                                    <div class="our-service__item-content">
                                        <?php if(!empty($item['service_title'])): ?>
                                            <h5 class="mb-15 text-uppercase">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo rrdevs_kses($item['service_title']); ?>
                                                </a>
                                            </h5>
                                        <?php endif; ?>
                                        <?php if(!empty($item['service_description'])): ?>
                                            <p class="mb-30"><?php echo rrdevs_kses($item['service_description']); ?></p>
                                        <?php endif; ?>

                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="rr-btn rr-btn__red">
                                            <span class="hover-rl"></span>
                                            <span class="fake_hover"></span>
                                            <span class="btn-wrap">
                                                <span class="text-one">
                                                    <?php echo rrdevs_kses($item['service_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                </span>
                                                <span class="text-two">
                                                    <?php echo rrdevs_kses($item['service_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                </span>
                                            </span>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

            <section class="delishs-el-section our-service-2 section-space">
                <div class="container">
                    <div class="row g-24">
                        <div class="col-12">
                            <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                    <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                        <?php print rrdevs_kses($settings['section_subheading']); ?>
                                    </span>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                    <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                        <?php print rrdevs_kses($settings['section_title']); ?>
                                    </h2>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                    <div class="dot-shpae wow clip-a-z">
                                        <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="row g-24 mb-minus-40">
                        <?php foreach ($settings['service_list_two'] as $index => $item) : 
                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }
                            ?>
                            <div class="col-lg-3 col-md-4 col-sm-6">
                                <div class="our-service-2__item mb-40">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="our-service-2__item-media">
                                        <img src="<?php print esc_url($service_image); ?>" alt="<?php print esc_attr($service_image_alt); ?>" class="img-fluid">
                                    </a>
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="btn">
                                        <?php echo rrdevs_kses($item['service_button_text']); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-3' ): ?>

            <section class="delishs-el-section service-2 section-space section-bg-3">
                <div class="container">
                    <div class="row g-24 mb-minus-24">
                        <?php foreach ($settings['service_list'] as $index => $item) : 
                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }
                            
                        ?>
                        <div class="col-xl-3 col-lg-4 col-sm-6">
                            <div class="service-2__item mb-24">
                                <?php if ( !empty( $service_image ) ) : ?>
                                    <div class="service-2__item-media">
                                        <img src="<?php print esc_url($service_image); ?>" class="img-fluid" alt="<?php print esc_attr($service_image_alt); ?>">
                                    </div>
                                <?php endif; ?>
                                <div class="service-2__item-text">
                                    <?php if ( !empty( $item['service_title'] ) ) : ?>
                                        <h6 class="text-uppercase"><?php print rrdevs_kses($item['service_title']); ?></h6>
                                    <?php endif; ?>
                                    <?php if ( !empty( $item['service_description'] ) ) : ?>
                                        <p><?php print rrdevs_kses($item['service_description']); ?></p>
                                    <?php endif; ?>
                                    <?php if ( !empty( $item['service_button_text'] ) ) : ?>
                                        <a class="btn-book" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php print rrdevs_kses($item['service_button_text']); ?>
                                        </a>
                                    <?php endif; ?>
                                    
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php endif; 
    }
}

$widgets_manager->register( new Delishs_Services() );