<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Hero extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_hero';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Hero', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

    public function get_delishs_contact_form(){
        if ( ! class_exists( 'WPCF7' ) ) {
            return;
        }
        $delishs_cfa         = array();
        $delishs_cf_args     = array( 'posts_per_page' => -1, 'post_type'=> 'wpcf7_contact_form' );
        $delishs_forms       = get_posts( $delishs_cf_args );
        $delishs_cfa         = ['0' => esc_html__( 'Select Form', 'delishs-core' ) ];
        if( $delishs_forms ){
            foreach ( $delishs_forms as $delishs_form ){
                $delishs_cfa[$delishs_form->ID] = $delishs_form->post_title;
            }
        }else{
            $delishs_cfa[ esc_html__( 'No contact form found', 'delishs-core' ) ] = 0;
        }
        return $delishs_cfa;
    }

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                    'layout-2' => esc_html__('Layout 2', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_control(
            'hero_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__('Title & Content', 'delishs-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'hero_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'WELCOME TO OUR AGENCY', 'delishs-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_control(
            'hero_title_one',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Lunch &', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'hero_title_two',
            [
                'label' => esc_html__( 'Title Two', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Happy Hour', 'delishs-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'hero_description',
            [
                'label' => esc_html__('Description', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Check our latest seasonal promotions', 'delishs-core'),
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__('Button', 'delishs-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'hero_button_text',
            [
                'label' => esc_html__( 'Button Text', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Book A Table', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'hero_button_link_type',
            [
                'label' => esc_html__('Button Link Type', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'hero_button_link',
            [
                'label' => esc_html__('Button link', 'delishs-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'label_block' => true,
                'condition' => [
                    'hero_button_link_type' => '1',
                ],
            ]
        );

        $this->add_control(
            'hero_button_page_link',
            [
                'label' => esc_html__('Select Button Page', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'hero_button_link_type' => '2',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'delishs_contact',
            [
                'label' => esc_html__('Contact Form', 'delishs-core'),
            ]
        );

        $this->add_control(
            'form_title',
            [
                'label' => esc_html__( 'Form Title', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__( 'Book a Table', 'delishs-core' ),
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_control(
            'delishs_select_contact_form',
            [
                'label'   => esc_html__( 'Select Form', 'delishs-core' ),
                'type'    => Controls_Manager::SELECT,
                'default' => '0',
                'options' => $this->get_delishs_contact_form(),
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__media::before' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__subtitle' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            'subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__subtitle' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subheading_typography',
                'selector' => '{{WRAPPER}} .banner-3__subtitle',
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
                
            ]
        );

        $this->add_control(
            '_heading_style_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-2__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-2__title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .banner-2__title',
            ]
        );

        $this->add_control(
            '_heading_style_title_two',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title Two', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'title_color_two',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-2__title span' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography_two',
                'selector' => '{{WRAPPER}} .banner-2__title span',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        // Description
        $this->add_control(
            '_heading_style_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-2__content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-2__content p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .banner-2__content p',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_button',
			[
				'label' => __( 'Button', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-2 .btn-wrap .text-one' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-2 .fake_hover:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn:after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:after' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-2 .btn-wrap .text-two' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-2:before' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'delishs-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
    
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_contact_form',
			[
				'label' => __( 'Contact Form', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
			]
		);

        $this->add_responsive_control(
            'form_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'form_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'form_title_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table-title' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_input_label',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Input Label', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'input_label_bottom_spacing',
            [
                'label'     => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table label' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'input_label_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table label' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_input',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Input', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'input_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table__select select' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3__book-a-table__input .input-wrapper input' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'input_border_color',
            [
                'label'     => esc_html__( 'Border', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__book-a-table__select select' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3__book-a-table__input .input-wrapper input' => 'border-color: {{VALUE}}',
                ],
            ]
        );

		// Button
        $this->add_control(
            '_heading_style_contact_form_button',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Button', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

        $this->start_controls_tabs( '_tabs_contact_form_button' );
        
        $this->start_controls_tab(
            'contact_form_button_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'contact_form_button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-solid .btn-wrap .text-one'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'contact_form_button_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-solid' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'contact_form_button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

		$this->add_control(
            'contact_form_button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-solid .btn-wrap .text-two'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'contact_form_button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-solid:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'contact_form_button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .rr-btn-solid' => 'border-radius: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .rr-btn-solid:before' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['hero_image']['url']) ) {
            $hero_image = !empty($settings['hero_image']['id']) ? wp_get_attachment_image_url( $settings['hero_image']['id'], 'full') : $settings['hero_image']['url'];
            $hero_image_alt = get_post_meta($settings["hero_image"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

		<?php if ( $settings['design_style']  == 'layout-1' ): 
            
            if ('2' == $settings['hero_button_link_type']) {
                $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['hero_button_page_link']));
                $this->add_render_attribute('delishs-button-arg', 'target', '_self');
                $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn-2 wow clip-a-z');
            } else {
                if ( ! empty( $settings['hero_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['hero_button_link'] );
                    $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn-2 wow clip-a-z');
                }
            }
            
            ?>

            <section class="delishs-el-section banner-2 banner-2__space overflow-hidden" data-background="<?php print esc_url($hero_image); ?>">
                <div class="container">
                    <div class="row g-24">
                        <div class="col-12">
                            <div class="banner-2__content">
                                <h1 class="banner-2__title mb-20 mb-sm-15 mb-xs-10 title-animation">
                                    <?php print rrdevs_kses( $settings['hero_title_one' ] ); ?> <br>
                                    <?php if ( !empty( $settings['hero_title_two' ] ) ) : ?>
                                        <span><?php print rrdevs_kses( $settings['hero_title_two' ] ); ?></span>
                                    <?php endif; ?>
                                </h1>
                                <?php if ( !empty($settings['hero_description']) ) : ?>
                                    <p class="delishs-el-section-description mb-45 mb-sm-40 mb-xs-35">
                                        <?php echo rrdevs_kses( $settings['hero_description'] ); ?>
                                    </p>
                                <?php endif; ?>
                                <?php if (!empty($settings['hero_button_text'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                        <span class="hover-rl"></span>
                                        <span class="fake_hover"></span>
                                        <span class="btn-wrap">
                                            <span class="text-one"><?php echo $settings['hero_button_text']; ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right.svg" alt="not found"></span>
                                            <span class="text-two"><?php echo $settings['hero_button_text']; ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right.svg" alt="not found"></span>
                                        </span>
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ):

            // Link
            if ('2' == $settings['hero_button_link_type']) {
                $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['hero_button_page_link']));
                $this->add_render_attribute('delishs-button-arg', 'target', '_self');
                $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__white wow clip-a-z');
            } else {
                if ( ! empty( $settings['hero_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['hero_button_link'] );
                    $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__white wow clip-a-z');
                }
            }

        ?>

            <section class="delishs-el-section banner-3 theme-bg-secondary ">
                <div class="banner-3-main__space overflow-hidden">
                    <div class="container">
                        <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                            <div class="banner-3__shape wow clip-a-z">
                                <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/banner-3/shape.png" alt="icon not found">
                            </div>
                        <?php endif; ?>
                        <div class="row g-24">
                            <div class="col-md-7">
                                <div class="banner-3__content">
                                    <?php if ( !empty( $settings['hero_subheading' ] ) ) : ?>
                                        <span class="banner-3__subtitle text-uppercase mb-15">
                                            <?php print rrdevs_kses( $settings['hero_subheading' ] ); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['hero_title_one' ] ) ) : ?>
                                        <h1 class="banner-3__title text-uppercase title-animation mb-55 mb-sm-30 mb-xs-25">
                                            <?php print rrdevs_kses( $settings['hero_title_one' ] ); ?>
                                        </h1>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['hero_button_text'])) : ?>
                                        <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                            <span class="hover-rl"></span>
                                            <span class="fake_hover"></span>
                                            <span class="btn-wrap">
                                                <span class="text-one"><?php echo $settings['hero_button_text']; ?></span>
                                                <span class="text-two"><?php echo $settings['hero_button_text']; ?></span>
                                            </span>
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>

                            <div class="col-md-5">
                                <div class="banner-3__media wow clip-a-z">
                                    <img src="<?php print esc_url($hero_image); ?>" class="img-fluid" alt="<?php print esc_attr($hero_image_alt); ?>">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="container">
                    <div class="col-12">
                        <div class="banner-3__book-a-table">
                            <?php if ( !empty( $settings['form_title' ] ) ) : ?>
                                <h5 class="banner-3__book-a-table-title">
                                    <?php print rrdevs_kses( $settings['form_title' ] ); ?>
                                </h5>
                            <?php endif; ?>
                            <?php if( !empty($settings['delishs_select_contact_form']) ) : ?> 
                                <?php echo do_shortcode( '[contact-form-7  id="'.$settings['delishs_select_contact_form'].'"]' ); ?> 
                            <?php else : ?>
                                <?php echo '<div class="alert alert-info"><p class="m-0">' . __('Please Select contact form.', 'delishs-core' ). '</p></div>'; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </section>

            <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Delishs_Hero() );