<?php

namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Gallery_Slider extends Widget_Base {

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'delishs_gallery_slider';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return esc_html__( 'Gallery Slider', 'delishs-core' );
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'delishs-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return [ 'delishs_core' ];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends() {
        return [ 'delishs-gallery' ];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'delishs-core' ),
                ],
                'default' => 'layout-1',
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'VIEW OUR PROJECTS', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'OUR FOOD GALLERY', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_title_shape',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_gallery_slider',
            [
                'label' => esc_html__( 'Gallery Slider', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'gallery_image',
            [
                'label' => esc_html__('Upload Icon Image', 'delishs-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],

            ]
        );
        
        $this->add_control(
            'gallery_list',
            [
                'label' => esc_html__( 'Gallery List', 'delishs-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'gallery_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'gallery_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'gallery_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'gallery_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => esc_html__( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_arrow',
            [
                'label' => esc_html__( 'Arrow', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->start_controls_tabs( 'arrow_tabs' );
        
        $this->start_controls_tab(
            'arrow_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            'arrow_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project .our-project__slider__arrow-prev, .our-project .our-project__slider__arrow-next' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'arrow_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project .our-project__slider__arrow-prev, .our-project .our-project__slider__arrow-next' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'arrow_border',
                'selector' => '{{WRAPPER}} .our-project .our-project__slider__arrow-prev, .our-project .our-project__slider__arrow-next',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'arrow_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            'arrow_color_hover',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project .our-project__slider__arrow-prev:hover, .our-project .our-project__slider__arrow-next:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'arrow_background_hover',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project .our-project__slider__arrow-prev:hover, .our-project .our-project__slider__arrow-next:hover' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'arrow_border_hover',
                'selector' => '{{WRAPPER}} .our-project .our-project__slider__arrow-prev:hover, .our-project .our-project__slider__arrow-next:hover',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-heading .sub-heading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_title_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title-spacing' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_gallery',
            [
                'label' => esc_html__( 'Gallery Slider', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'gallery_image_background_hover',
            [
                'label' => esc_html__( 'Image Background (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project__item-hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .food-gallery__item-hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_gallery_icon',
            [
                'label' => esc_html__( 'Icon', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'gallery_icon_color',
            [
                'label' => esc_html__( 'Text', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project__item-hover-icon i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .food-gallery__item-hover-icon i' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'gallery_icon_background_color',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-project__item-hover-icon' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .food-gallery__item-hover-icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        

        ?>

        <?php if ( $settings['design_style']  == 'layout-1' ): ?>
        
            <section class="delishs-el-section our-project section-space__top">
                <div class="container">
                    <div class="delishs-el-section-title-spacing row g-24 mb-70 mb-sm-50 mb-xs-40 align-items-lg-end align-items-center">
                        <div class="col-xl-6">
                            <div class="section__title-wrapper text-center text-xl-start">
                                <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                    <span class="delishs-el-section-subheading section__subtitle justify-content-start mb-10 mb-xs-5 wow clip-a-z">
                                        <?php print rrdevs_kses($settings['section_subheading']); ?>
                                    </span>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                    <h2 class="delishs-el-section-title section__title mb-0 title-animation">
                                        <?php print rrdevs_kses($settings['section_title']); ?>
                                    </h2>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="col-xl-6">
                            <div class="d-flex justify-content-xl-end mt-xs-20 mt-sm-20 mt-md-20 mt-lg-20 justify-content-center">
                                <div class="our-project__slider__arrow d-flex justify-content-xl-end justify-content-center">
                                    <button class="our-project__slider__arrow-prev d-flex align-items-center justify-content-center wow clip-a-z">
                                        <i class="fa-solid fa-arrow-left"></i>
                                    </button>

                                    <button class="our-project__slider__arrow-next d-flex align-items-center justify-content-center wow clip-a-z">
                                        <i class="fa-solid fa-arrow-right"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="swiper our-project__slider">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['gallery_list'] as $item) : 

                            // Image
                            if ( !empty($item['gallery_image']['url']) ) {
                                $gallery_image = !empty($item['gallery_image']['id']) ? wp_get_attachment_image_url( $item['gallery_image']['id'], 'full') : $item['gallery_image']['url'];
                                $gallery_image_alt = get_post_meta($item["gallery_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            // Link
                            if ('2' == $item['gallery_button_link_type']) {
                                $link = get_permalink($item['gallery_button_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['gallery_button_link']['url']) ? $item['gallery_button_link']['url'] : '';
                                $target = !empty($item['gallery_button_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['gallery_button_link']['nofollow']) ? 'nofollow' : '';
                            }
                        ?>
                        <div class="swiper-slide">
                            <div class="our-project__item">
                                <div class="our-project__item-media">
                                    <img src="<?php echo esc_url($gallery_image); ?>" alt="<?php echo esc_attr($gallery_image_alt); ?>">
                                </div>
                                <div class="our-project__item-hover">
                                    <a href="<?php echo esc_url($gallery_image); ?>" class="our-project__item-hover-icon popup-image">
                                        <i class="fa-regular fa-eye"></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>
        
        <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

            <section class="delishs-el-section food-gallery section-space section-bg-4" data-background="<?php print get_template_directory_uri(); ?>/assets/imgs/food-gallery/bg.png">
                <div class="container">
                    <div class="row g-24">
                        <div class="col-12">
                            <div class="delishs-el-section-title-spacing section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                    <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                        <?php print rrdevs_kses($settings['section_subheading']); ?>
                                    </span>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                    <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                        <?php print rrdevs_kses($settings['section_title']); ?>
                                    </h2>
                                <?php endif; ?>
                                <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                    <div class="dot-shpae wow clip-a-z">
                                        <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="row g-24 mb-minus-24">
                        <?php foreach ($settings['gallery_list'] as $item) : 
                            // Image
                            if ( !empty($item['gallery_image']['url']) ) {
                                $gallery_image = !empty($item['gallery_image']['id']) ? wp_get_attachment_image_url( $item['gallery_image']['id'], 'full') : $item['gallery_image']['url'];
                                $gallery_image_alt = get_post_meta($item["gallery_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                        ?>
                        <div class="col-lg-4 col-md-6">
                            <div class="food-gallery__item mb-24">
                                <div class="food-gallery__item-media">
                                    <img src="<?php echo esc_url($gallery_image); ?>" alt="<?php echo esc_attr($gallery_image_alt); ?>">
                                </div>
                                <div class="food-gallery__item-hover">
                                    <a href="<?php echo esc_url($gallery_image); ?>" class="food-gallery__item-hover-icon popup-image">
                                        <i class="fa-regular fa-eye"></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>
        
        <?php endif; ?>

        <?php 
    }
}

$widgets_manager->register( new Delishs_Gallery_Slider() );