<?php
namespace DelishsCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delishs Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Food_Menu extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_food_menu';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Food Menu', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'delishs-core' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Make Reservation', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Discover Our Category', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title_shape',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_food_menu_left',
            [
                'label' => esc_html__( 'Left Food Menu',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'food_menu_left_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'BEST DRINKS', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'food_menu_left_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Starter Menu', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $repeater = new Repeater();

        $repeater->add_control(
            'food_menu_left_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $repeater->add_control(
            'food_menu_left_food_name',
            [
                'label' => esc_html__( 'Food Name', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_left_food_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_left_food_currency',
            [
                'label' => __('Currency', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'delishs-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'delishs-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'delishs-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'delishs-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'delishs-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'delishs-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'delishs-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'delishs-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'delishs-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'delishs-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'delishs-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'delishs-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'delishs-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'delishs-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'delishs-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'delishs-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'delishs-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'delishs-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'delishs-core'),
                    'custom' => __('Custom', 'delishs-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $repeater->add_control(
            'food_menu_left_food_currency_custom',
            [
                'label' => __('Custom Symbol', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'food_menu_left_food_price',
            [
                'label' => esc_html__('Price', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true
                ]
            ]
        );
        
        $repeater->add_control(
            'food_menu_left_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_left_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'food_menu_left_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_left_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'food_menu_left_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_control(
            'food_menu_left_list',
            [
                'label' => esc_html__( 'Food List', 'delishs-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'food_menu_left_food_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_left_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_left_food_price' => __( '16.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_left_food_name' => __( 'Butternut Squash Soup', 'delishs-core' ),
                        'food_menu_left_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_left_food_price' => __( '25.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_left_food_name' => __( 'Braised Short Ribs', 'delishs-core' ),
                        'food_menu_left_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_left_food_price' => __( '11.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_left_food_name' => __( 'Braised Long Ribs', 'delishs-core' ),
                        'food_menu_left_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_left_food_price' => __( '14.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_left_food_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_left_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_left_food_price' => __( '19.00', 'delishs-core' ),
                    ],
                ],
                'title_field' => '{{{ food_menu_left_food_name }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_food_menu_right',
            [
                'label' => esc_html__( 'Right Food Menu',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'food_menu_right_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Best Foods', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'food_menu_right_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Main Dishes', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $repeater = new Repeater();

        $repeater->add_control(
            'food_menu_right_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $repeater->add_control(
            'food_menu_right_food_name',
            [
                'label' => esc_html__( 'Food Name', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_right_food_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_right_food_currency',
            [
                'label' => __('Currency', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'delishs-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'delishs-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'delishs-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'delishs-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'delishs-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'delishs-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'delishs-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'delishs-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'delishs-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'delishs-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'delishs-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'delishs-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'delishs-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'delishs-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'delishs-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'delishs-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'delishs-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'delishs-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'delishs-core'),
                    'custom' => __('Custom', 'delishs-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $repeater->add_control(
            'food_menu_right_food_currency_custom',
            [
                'label' => __('Custom Symbol', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'food_menu_right_food_price',
            [
                'label' => esc_html__('Price', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true
                ]
            ]
        );
        
        $repeater->add_control(
            'food_menu_right_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_right_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'food_menu_right_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_right_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'food_menu_right_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_control(
            'food_menu_right_list',
            [
                'label' => esc_html__( 'Food List', 'delishs-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'food_menu_right_food_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_right_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_right_food_price' => __( '16.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_right_food_name' => __( 'Butternut Squash Soup', 'delishs-core' ),
                        'food_menu_right_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_right_food_price' => __( '25.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_right_food_name' => __( 'Braised Short Ribs', 'delishs-core' ),
                        'food_menu_right_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_right_food_price' => __( '11.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_right_food_name' => __( 'Braised Long Ribs', 'delishs-core' ),
                        'food_menu_right_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_right_food_price' => __( '14.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_right_food_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_right_food_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_right_food_price' => __( '19.00', 'delishs-core' ),
                    ],
                ],
                'title_field' => '{{{ food_menu_right_food_name }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_design_layout_inner',
            [
                'label' => esc_html__( 'Inner', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'design_layout_inner_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__background' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_inner_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__background' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_inner_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__background' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .our-food-menu-list__item-text h6' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .our-food-menu-list__item span' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_layout_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .section__title-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_food_menu',
            [
                'label' => esc_html__( 'Food Menu', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_food_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'food_subheading_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__heading-title .bar span' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__heading-title .bar span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_subheading_shape_color',
            [
                'label' => esc_html__( 'Shape Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__heading-title .bar span:after' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .our-food-menu__heading-title .bar span:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_subheading_typography',
                'selector' => '{{WRAPPER}} .our-food-menu__heading-title .bar span',
            ]
        );

        $this->add_control(
            '_heading_style_food_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__heading-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu__heading-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_title_typography',
                'selector' => '{{WRAPPER}} .our-food-menu__heading-title',
            ]
        );

        $this->add_control(
            '_heading_style_food_name',
            [
                'label' => esc_html__( 'Name', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_name_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_name_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text h6' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_name_color_hover',
            [
                'label' => esc_html__( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text h6 a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_name_typography',
                'selector' => '{{WRAPPER}} .reservation-2__item-text h6',
            ]
        );

        $this->add_control(
            '_heading_style_food_price',
            [
                'label' => esc_html__( 'Price', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'food_price_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text::before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text::before' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_price_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_price_typography',
                'selector' => '{{WRAPPER}} .our-food-menu-list__item span, .reservation-2__item span',
            ]
        );

        $this->add_control(
            '_heading_style_food_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_description_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .reservation-2__item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_description_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_description_typography',
                'selector' => '{{WRAPPER}} .our-food-menu-list__item p, .reservation-2__item p',
            ]
        );

        $this->add_control(
            '_heading_style_food_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .reservation-2__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .reservation-2__item' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item span' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

	}

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }


	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>
            
                <section class="delishs-el-section our-food-menu our-food-menu__space our-food-menu__top-up our-food-menu__after-background position-relative overflow-hidden">
                    <div class="container">
                        <div class="our-food-menu__background section-space">
                            <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                                <div class="our-food-menu__shape">
                                    <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/our-food-menu/shape.png" alt="not found">
                                </div>
                            <?php endif; ?>

                            <div class="row g-24">
                                <div class="col-12">
                                    <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                        <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                            <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                                <?php print rrdevs_kses($settings['section_subheading']); ?>
                                            </span>
                                        <?php endif; ?>
                                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                            <h2 class="delishs-el-section-title section__title section__title-white mb-10 mb-xs-5 title-animation">
                                                <?php print rrdevs_kses($settings['section_title']); ?>
                                            </h2>
                                        <?php endif; ?>
                                        <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                            <div class="dot-shpae wow clip-a-z">
                                                <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>

                            <div class="our-food-menu-wrapper">
                                <div class="our-food-menu__heading mb-35 d-none d-xl-flex">
                                    <h5 class="our-food-menu__heading-title text-uppercase">
                                        <?php if ( !empty( $settings['food_menu_left_subheading'] ) ) : ?>
                                            <span class="bar">
                                                <span><?php print rrdevs_kses($settings['food_menu_left_subheading']); ?></span>
                                            </span>
                                        <?php endif; ?>
                                        <?php print rrdevs_kses($settings['food_menu_left_title']); ?>
                                    </h5>

                                    <h5 class="our-food-menu__heading-title text-uppercase">
                                        <?php if ( !empty( $settings['food_menu_right_subheading'] ) ) : ?>
                                            <span class="bar"><span><?php print rrdevs_kses($settings['food_menu_right_subheading']); ?></span></span>
                                        <?php endif; ?>
                                        <?php print rrdevs_kses($settings['food_menu_right_title']); ?>
                                    </h5>
                                </div>
                                <div class="our-food-menu-list">
                                    <div class="our-food-menu-list__left mb-md-50 mb-lg-50 mb-sm-50 mb-xs-50">
                                        <h5 class="our-food-menu__heading-title text-uppercase mb-30 d-flex d-xl-none">
                                            <?php if ( !empty( $settings['food_menu_left_subheading'] ) ) : ?>
                                                <span class="bar">
                                                    <span><?php print rrdevs_kses($settings['food_menu_left_subheading']); ?></span>
                                                </span>
                                            <?php endif; ?>
                                            <?php print rrdevs_kses($settings['food_menu_left_title']); ?>
                                        </h5>

                                        <?php foreach ($settings['food_menu_left_list'] as $item) :

                                            if ( !empty($item['food_menu_left_image']['url']) ) {
                                                $food_menu_left_image = !empty($item['food_menu_left_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_left_image']['id'], 'full') : $item['food_menu_left_image']['url'];
                                                $food_menu_left_image_alt = get_post_meta($item["food_menu_left_image"]["id"], "_wp_attachment_image_alt", true);
                                            }

                                            if ($item['food_menu_left_food_currency'] === 'custom') {
                                                $left_currency = $item['food_menu_left_food_currency_custom'];
                                            } else {
                                                $left_currency = self::get_currency_symbol($item['food_menu_left_food_currency']);
                                            }

                                            if ('2' == $item['food_menu_left_button_link_type']) {
                                                $link = get_permalink($item['food_menu_left_button_page_link']);
                                                $target = '_self';
                                                $rel = 'nofollow';
                                            } else {
                                                $link = !empty($item['food_menu_left_button_link']['url']) ? $item['food_menu_left_button_link']['url'] : '';
                                                $target = !empty($item['food_menu_left_button_link']['is_external']) ? '_blank' : '';
                                                $rel = !empty($item['food_menu_left_button_link']['nofollow']) ? 'nofollow' : '';
                                            }
                                                            
                                        ?>
                                        <div class="our-food-menu-list__item">
                                            <div class="our-food-menu-list__item-media">
                                                <img src="<?php print esc_url($food_menu_left_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_left_image_alt); ?>">
                                            </div>
                                            <div class="our-food-menu-list__item-wrapper">
                                                <div class="our-food-menu-list__item-text">
                                                    <h6>
                                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                            <?php print rrdevs_kses($item['food_menu_left_food_name']); ?>
                                                        </a>
                                                    </h6>
                                                    <span><?php echo esc_attr($left_currency); ?><?php print rrdevs_kses($item['food_menu_left_food_price']); ?></span>
                                                </div>
                                                <p><?php print rrdevs_kses($item['food_menu_left_food_description']); ?></p>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>

                                    <div class="our-food-menu-list__right">
                                        <h5 class="our-food-menu__heading-title text-uppercase mb-30 d-flex d-xl-none">
                                            <?php if ( !empty( $settings['food_menu_right_subheading'] ) ) : ?>
                                                <span class="bar"><span><?php print rrdevs_kses($settings['food_menu_right_subheading']); ?></span></span>
                                            <?php endif; ?>
                                            <?php print rrdevs_kses($settings['food_menu_right_title']); ?>
                                        </h5>

                                        <?php foreach ($settings['food_menu_right_list'] as $item) :

                                            if ( !empty($item['food_menu_right_image']['url']) ) {
                                                $food_menu_right_image = !empty($item['food_menu_right_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_right_image']['id'], 'full') : $item['food_menu_right_image']['url'];
                                                $food_menu_right_image_alt = get_post_meta($item["food_menu_right_image"]["id"], "_wp_attachment_image_alt", true);
                                            }

                                            if ($item['food_menu_right_food_currency'] === 'custom') {
                                                $right_currency = $item['food_menu_right_food_currency_custom'];
                                            } else {
                                                $right_currency = self::get_currency_symbol($item['food_menu_right_food_currency']);
                                            }

                                            if ('2' == $item['food_menu_right_button_link_type']) {
                                                $link = get_permalink($item['food_menu_right_button_page_link']);
                                                $target = '_self';
                                                $rel = 'nofollow';
                                            } else {
                                                $link = !empty($item['food_menu_right_button_link']['url']) ? $item['food_menu_right_button_link']['url'] : '';
                                                $target = !empty($item['food_menu_right_button_link']['is_external']) ? '_blank' : '';
                                                $rel = !empty($item['food_menu_right_button_link']['nofollow']) ? 'nofollow' : '';
                                            }
                                                            
                                        ?>
                                        <div class="our-food-menu-list__item">
                                            <div class="our-food-menu-list__item-media">
                                                <img src="<?php print esc_url($food_menu_right_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_right_image_alt); ?>">
                                            </div>
                                            <div class="our-food-menu-list__item-wrapper">
                                                <div class="our-food-menu-list__item-text">
                                                    <h6>
                                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                            <?php print rrdevs_kses($item['food_menu_right_food_name']); ?>
                                                        </a>
                                                    </h6>
                                                    <span><?php echo esc_attr($right_currency); ?><?php print rrdevs_kses($item['food_menu_right_food_price']); ?></span>
                                                </div>
                                                <p><?php print rrdevs_kses($item['food_menu_right_food_description']); ?></p>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>
            
                <section class="delishs-el-section reservation-2 section-space section-bg-2 overflow-hidden">
                    <div class="container">
                        <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                            <div class="reservation-2__shape">
                                <img class="image-1 leftRight" src="<?php print get_template_directory_uri(); ?>/assets/imgs/reservation-2/shap-1.svg" alt="not found">
                                <img class="image-2 rightLeft" src="<?php print get_template_directory_uri(); ?>/assets/imgs/reservation-2/shap-2.svg" alt="not found">
                            </div>
                        <?php endif; ?>

                        <div class="row g-24">
                            <div class="col-12">
                                <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                    <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                            <?php print rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                            <?php print rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                        <div class="dot-shpae wow clip-a-z">
                                            <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24 mb-minus-20">
                            <div class="col-xl-6">
                                <?php foreach ($settings['food_menu_left_list'] as $item) :

                                    if ( !empty($item['food_menu_left_image']['url']) ) {
                                        $food_menu_left_image = !empty($item['food_menu_left_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_left_image']['id'], 'full') : $item['food_menu_left_image']['url'];
                                        $food_menu_left_image_alt = get_post_meta($item["food_menu_left_image"]["id"], "_wp_attachment_image_alt", true);
                                    }

                                    if ($item['food_menu_left_food_currency'] === 'custom') {
                                        $left_currency = $item['food_menu_left_food_currency_custom'];
                                    } else {
                                        $left_currency = self::get_currency_symbol($item['food_menu_left_food_currency']);
                                    }

                                    if ('2' == $item['food_menu_left_button_link_type']) {
                                        $link = get_permalink($item['food_menu_left_button_page_link']);
                                        $target = '_self';
                                        $rel = 'nofollow';
                                    } else {
                                        $link = !empty($item['food_menu_left_button_link']['url']) ? $item['food_menu_left_button_link']['url'] : '';
                                        $target = !empty($item['food_menu_left_button_link']['is_external']) ? '_blank' : '';
                                        $rel = !empty($item['food_menu_left_button_link']['nofollow']) ? 'nofollow' : '';
                                    }
                                                
                                ?>
                                <div class="reservation-2__item">
                                    <div class="reservation-2__item-media">
                                        <img src="<?php print esc_url($food_menu_left_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_left_image_alt); ?>">
                                    </div>
                                    <div class="reservation-2__item-wrapper">
                                        <div class="reservation-2__item-text">
                                            <h6>
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php print rrdevs_kses($item['food_menu_left_food_name']); ?>
                                                </a>
                                            </h6>
                                            <span><?php echo esc_attr($left_currency); ?><?php print rrdevs_kses($item['food_menu_left_food_price']); ?></span>
                                        </div>
                                        <p><?php print rrdevs_kses($item['food_menu_left_food_description']); ?></p>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="col-xl-6">
                                <?php foreach ($settings['food_menu_right_list'] as $item) :

                                    if ( !empty($item['food_menu_right_image']['url']) ) {
                                        $food_menu_right_image = !empty($item['food_menu_right_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_right_image']['id'], 'full') : $item['food_menu_right_image']['url'];
                                        $food_menu_right_image_alt = get_post_meta($item["food_menu_right_image"]["id"], "_wp_attachment_image_alt", true);
                                    }

                                    if ($item['food_menu_right_food_currency'] === 'custom') {
                                        $right_currency = $item['food_menu_right_food_currency_custom'];
                                    } else {
                                        $right_currency = self::get_currency_symbol($item['food_menu_right_food_currency']);
                                    }

                                    if ('2' == $item['food_menu_right_button_link_type']) {
                                        $link = get_permalink($item['food_menu_right_button_page_link']);
                                        $target = '_self';
                                        $rel = 'nofollow';
                                    } else {
                                        $link = !empty($item['food_menu_right_button_link']['url']) ? $item['food_menu_right_button_link']['url'] : '';
                                        $target = !empty($item['food_menu_right_button_link']['is_external']) ? '_blank' : '';
                                        $rel = !empty($item['food_menu_right_button_link']['nofollow']) ? 'nofollow' : '';
                                    }
                                                
                                ?>
                                <div class="reservation-2__item">
                                    <div class="reservation-2__item-media">
                                        <img src="<?php print esc_url($food_menu_right_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_right_image_alt); ?>">
                                    </div>
                                    <div class="reservation-2__item-wrapper">
                                        <div class="reservation-2__item-text">
                                            <h6>
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php print rrdevs_kses($item['food_menu_right_food_name']); ?>
                                                </a>
                                            </h6>
                                            <span><?php echo esc_attr($right_currency); ?><?php print rrdevs_kses($item['food_menu_right_food_price']); ?></span>
                                        </div>
                                        <p><?php print rrdevs_kses($item['food_menu_right_food_description']); ?></p>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>    
            
            <?php endif; ?>

        <?php
	}

}

$widgets_manager->register( new Delishs_Food_Menu() );