<?php
namespace DelishsCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delishs Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Food_Menu_Two extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_food_menu_two';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Food Menu Two', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_food_menu',
            [
                'label' => esc_html__( 'Food Menu',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'delishs-core' ),
                'type' => Controls_Manager::HIDDEN,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'delishs-core' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'food_menu_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $this->add_control(
            'food_menu_name',
            [
                'label' => esc_html__( 'Food Name', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Wild Mushroom Arancini', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'food_menu_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'food_menu_currency',
            [
                'label' => __('Currency', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'delishs-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'delishs-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'delishs-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'delishs-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'delishs-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'delishs-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'delishs-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'delishs-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'delishs-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'delishs-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'delishs-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'delishs-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'delishs-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'delishs-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'delishs-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'delishs-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'delishs-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'delishs-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'delishs-core'),
                    'custom' => __('Custom', 'delishs-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $this->add_control(
            'food_menu_currency_custom',
            [
                'label' => __('Custom Symbol', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'food_menu_price',
            [
                'label' => esc_html__('Price', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __( '18.00', 'delishs-core' ),
                'dynamic' => [
                    'active' => true
                ]
            ]
        );
        
        $this->add_control(
            'food_menu_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'food_menu_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'food_menu_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'food_menu_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'food_menu_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .reservation-2__item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .reservation-2__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .reservation-2__item' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item span' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_food_menu',
            [
                'label' => esc_html__( 'Food Menu', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_food_name',
            [
                'label' => esc_html__( 'Name', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_name_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_name_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text h6' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_name_color_hover',
            [
                'label' => esc_html__( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text h6 a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_name_typography',
                'selector' => '{{WRAPPER}} .reservation-2__item-text h6',
            ]
        );

        $this->add_control(
            '_heading_style_food_price',
            [
                'label' => esc_html__( 'Price', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'food_price_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item-text::before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item-text::before' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_price_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_price_typography',
                'selector' => '{{WRAPPER}} .our-food-menu-list__item span, .reservation-2__item span',
            ]
        );

        $this->add_control(
            '_heading_style_food_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_description_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .reservation-2__item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_description_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-food-menu-list__item p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .reservation-2__item p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_description_typography',
                'selector' => '{{WRAPPER}} .our-food-menu-list__item p, .reservation-2__item p',
            ]
        );

        $this->end_controls_section();

	}

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }


	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['food_menu_image']['url']) ) {
            $food_menu_image = !empty($settings['food_menu_image']['id']) ? wp_get_attachment_image_url( $settings['food_menu_image']['id'], 'full') : $settings['food_menu_image']['url'];
            $food_menu_image_alt = get_post_meta($settings["food_menu_image"]["id"], "_wp_attachment_image_alt", true);
        }

        if ($settings['food_menu_currency'] === 'custom') {
            $currency = $settings['food_menu_currency_custom'];
        } else {
            $currency = self::get_currency_symbol($settings['food_menu_currency']);
        }

        if ('2' == $settings['food_menu_button_link_type']) {
            $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['food_menu_button_page_link']));
            $this->add_render_attribute('delishs-button-arg', 'target', '_self');
            $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('delishs-button-arg', 'class', '');
        } else {
            if ( ! empty( $settings['food_menu_button_link']['url'] ) ) {
                $this->add_link_attributes( 'delishs-button-arg', $settings['food_menu_button_link'] );
                $this->add_render_attribute('delishs-button-arg', 'class', '');
            }
        }

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>

                <div class="reservation-2__item">
                    <div class="reservation-2__item-media">
                        <img src="<?php print esc_url($food_menu_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_image_alt); ?>">
                    </div>
                    <div class="reservation-2__item-wrapper">
                        <div class="reservation-2__item-text">
                            <h6>
                                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                    <?php print rrdevs_kses($settings['food_menu_name']); ?>
                                </a>
                            </h6>
                            <span><?php echo esc_attr($currency); ?><?php print rrdevs_kses($settings['food_menu_price']); ?></span>
                        </div>
                        <p><?php print rrdevs_kses($settings['food_menu_description']); ?></p>
                    </div>
                </div>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>
            
            <?php endif; ?>

        <?php
	}

}

$widgets_manager->register( new Delishs_Food_Menu_Two() );