<?php
namespace DelishsCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delishs Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Food_Menu_List extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_food_menu_list';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Food Menu List', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'delishs-core' ),
                    'layout-3' => esc_html__( 'Layout 3', 'delishs-core' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'section_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_food_menu',
            [
                'label' => esc_html__( 'Food Menu',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $repeater = new Repeater();
        
        $repeater->add_control(
            'food_menu_name',
            [
                'label' => esc_html__( 'Food Name', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'currency',
            [
                'label' => __('Currency', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'delishs-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'delishs-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'delishs-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'delishs-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'delishs-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'delishs-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'delishs-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'delishs-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'delishs-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'delishs-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'delishs-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'delishs-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'delishs-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'delishs-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'delishs-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'delishs-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'delishs-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'delishs-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'delishs-core'),
                    'custom' => __('Custom', 'delishs-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $repeater->add_control(
            'currency_custom',
            [
                'label' => __('Custom Symbol', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'food_menu_price',
            [
                'label' => esc_html__( 'Price', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );
        
        $repeater->add_control(
            'food_menu_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'food_menu_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'food_menu_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_control(
            'food_menu_list',
            [
                'label' => esc_html__( 'Food List', 'delishs-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'food_menu_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_price' => __( '16.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Butternut Squash Soup', 'delishs-core' ),
                        'food_menu_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_price' => __( '25.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Braised Short Ribs', 'delishs-core' ),
                        'food_menu_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_price' => __( '11.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Braised Long Ribs', 'delishs-core' ),
                        'food_menu_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_price' => __( '14.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_description' => __( 'Ricotta, goat cheese, beetroot and dateline.', 'delishs-core' ),
                        'food_menu_price' => __( '19.00', 'delishs-core' ),
                    ],
                ],
                'title_field' => '{{{ food_menu_name }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_food_menu_two',
            [
                'label' => esc_html__( 'Food Menu',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $repeater = new Repeater();

        $repeater->add_control(
            'food_menu_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $repeater->add_control(
            'food_menu_name',
            [
                'label' => esc_html__( 'Food Name', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'currency',
            [
                'label' => __('Currency', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'delishs-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'delishs-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'delishs-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'delishs-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'delishs-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'delishs-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'delishs-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'delishs-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'delishs-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'delishs-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'delishs-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'delishs-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'delishs-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'delishs-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'delishs-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'delishs-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'delishs-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'delishs-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'delishs-core'),
                    'custom' => __('Custom', 'delishs-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $repeater->add_control(
            'currency_custom',
            [
                'label' => __('Custom Symbol', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'food_menu_price',
            [
                'label' => esc_html__( 'Price', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'food_rating',
            [
                'label' => esc_html__( 'Rating', 'delishs-core' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 5,
                'step' => .5,
                'default' => 5,
            ]
        );
        
        $repeater->add_control(
            'food_menu_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'food_menu_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'food_menu_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'food_menu_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'food_menu_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_control(
            'food_menu_list_two',
            [
                'label' => esc_html__( 'Food List', 'delishs-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'food_menu_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_price' => __( '16.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Butternut Squash Soup', 'delishs-core' ),
                        'food_menu_price' => __( '25.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Braised Short Ribs', 'delishs-core' ),
                        'food_menu_price' => __( '11.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Braised Long Ribs', 'delishs-core' ),
                        'food_menu_price' => __( '14.00', 'delishs-core' ),
                    ],
                    [
                        'food_menu_name' => __( 'Mushroom Arancini', 'delishs-core' ),
                        'food_menu_price' => __( '19.00', 'delishs-core' ),
                    ],
                ],
                'title_field' => '{{{ food_menu_name }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_column',
            [
                'label' => esc_html__( 'Column',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'delishs-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_food_menu',
            [
                'label' => esc_html__( 'Food Menu', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_food_name',
            [
                'label' => esc_html__( 'Name', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_name_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .text h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .food-menu__item-text h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'food_name_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .text h6' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item-text h6' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_name_color_hover',
            [
                'label' => esc_html__( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .text h6 a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item-text h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_name_typography',
                'selector' => '{{WRAPPER}} .food-menu__price-nav__list li .text h6, .food-menu__item-text h6',
            ]
        );

        $this->add_control(
            '_heading_style_food_price',
            [
                'label' => esc_html__( 'Price', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'food_price_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .price' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item-text .price' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_price_typography',
                'selector' => '{{WRAPPER}} .food-menu__price-nav__list li .price, .food-menu__item-text .price',
            ]
        );

        $this->add_control(
            '_heading_style_food_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_responsive_control(
            'food_description_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .text p' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'food_description_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li .text p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_description_typography',
                'selector' => '{{WRAPPER}} .food-menu__price-nav__list li .text p',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_food_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .food-menu__item-text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'food_layout_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'food_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list li:not(:last-child)' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item-text' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'food_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .food-menu__price-nav__list' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .food-menu__item-text' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

	}

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }


	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['section_image']['url']) ) {
            $section_image = !empty($settings['section_image']['id']) ? wp_get_attachment_image_url( $settings['section_image']['id'], 'full') : $settings['section_image']['url'];
            $section_image_alt = get_post_meta($settings["section_image"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>

                <div class="delishs-el-section food-menu__price-nav">
                    <div class="food-menu__price-nav__media">
                        <img src="<?php print esc_url($section_image); ?>" class="img-fluid" alt="<?php print esc_attr($section_image_alt); ?>">
                    </div>

                    <ul class="food-menu__price-nav__list">
                        <?php foreach ($settings['food_menu_list'] as $item) :

                            if ($item['currency'] === 'custom') {
                                $currency = $item['currency_custom'];
                            } else {
                                $currency = self::get_currency_symbol($item['currency']);
                            }

                            if ('2' == $item['food_menu_button_link_type']) {
                                $link = get_permalink($item['food_menu_button_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['food_menu_button_link']['url']) ? $item['food_menu_button_link']['url'] : '';
                                $target = !empty($item['food_menu_button_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['food_menu_button_link']['nofollow']) ? 'nofollow' : '';
                            }
                                            
                        ?>
                        <li>
                            <div class="text">
                                <h6>
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                        <?php print rrdevs_kses($item['food_menu_name']); ?>
                                    </a>
                                </h6>
                                <p class="mb-0"><?php print rrdevs_kses($item['food_menu_description']); ?></p>
                            </div>
                            <span class="price">
                                <?php echo esc_attr($currency); ?><?php print rrdevs_kses($item['food_menu_price']); ?>
                            </span>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <div class="delishs-el-section food-menu__item-wrapper">
                    <div class="row g-10 grid">
                        <?php foreach ($settings['food_menu_list_two'] as $item) :
                        
                            if ( !empty($item['food_menu_image']['url']) ) {
                                $food_menu_image = !empty($item['food_menu_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_image']['id'], 'full') : $item['food_menu_image']['url'];
                                $food_menu_image_alt = get_post_meta($item["food_menu_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            if ($item['currency'] === 'custom') {
                                $currency = $item['currency_custom'];
                            } else {
                                $currency = self::get_currency_symbol($item['currency']);
                            }

                            if ('2' == $item['food_menu_button_link_type']) {
                                $link = get_permalink($item['food_menu_button_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['food_menu_button_link']['url']) ? $item['food_menu_button_link']['url'] : '';
                                $target = !empty($item['food_menu_button_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['food_menu_button_link']['nofollow']) ? 'nofollow' : '';
                            }
                                            
                        ?>
                        <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> grid-item breakfast">
                            <div class="food-menu__item">
                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="food-menu__item-media">
                                    <img src="<?php print esc_url($food_menu_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_image_alt); ?>">
                                </a>

                                <div class="food-menu__item-text">
                                    <ul class="rating">
                                        <?php 
                                        $food_rating = $item['food_rating'];

                                        // Calculate full stars, half stars, and empty stars
                                        $full_stars = floor($food_rating);
                                        $half_star = ($food_rating - $full_stars >= 0.5) ? 1 : 0;
                                        $empty_stars = 5 - ($full_stars + $half_star);
                                        for ($i = 1; $i <= $full_stars; $i++): ?>
                                            <li><i class="fa-solid fa-star"></i></li>
                                        <?php endfor; ?>

                                        <?php if ($half_star): ?>
                                            <li><i class="fa-solid fa-star-half-stroke"></i></li>
                                        <?php endif; ?>

                                        <?php for ($i = 1; $i <= $empty_stars; $i++): ?>
                                            <li><i class="fa-regular fa-star"></i></li>
                                        <?php endfor; ?>
                                    </ul>
                                    <h6>
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php print rrdevs_kses($item['food_menu_name']); ?>
                                        </a>
                                    </h6>
                                    <span class="price">
                                        <?php echo esc_attr($currency); ?><?php print rrdevs_kses($item['food_menu_price']); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>

            <?php elseif ( $settings['design_style']  == 'layout-3' ): ?>

                <div class="row g-24 mb-minus-24">
                    <?php foreach ($settings['food_menu_list_two'] as $item) :
                        
                        if ( !empty($item['food_menu_image']['url']) ) {
                            $food_menu_image = !empty($item['food_menu_image']['id']) ? wp_get_attachment_image_url( $item['food_menu_image']['id'], 'full') : $item['food_menu_image']['url'];
                            $food_menu_image_alt = get_post_meta($item["food_menu_image"]["id"], "_wp_attachment_image_alt", true);
                        }

                        if ($item['currency'] === 'custom') {
                            $currency = $item['currency_custom'];
                        } else {
                            $currency = self::get_currency_symbol($item['currency']);
                        }

                        if ('2' == $item['food_menu_button_link_type']) {
                            $link = get_permalink($item['food_menu_button_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['food_menu_button_link']['url']) ? $item['food_menu_button_link']['url'] : '';
                            $target = !empty($item['food_menu_button_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['food_menu_button_link']['nofollow']) ? 'nofollow' : '';
                        }
                                        
                    ?>  
                        <div class="col-xl-3 col-md-4 col-sm-6">
                            <div class="food-menu-tab__item mb-24">
                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="food-menu-tab__item-media">
                                    <img src="<?php print esc_url($food_menu_image); ?>" class="img-fluid" alt="<?php print esc_attr($food_menu_image_alt); ?>">
                                </a>
                                <div class="food-menu-tab__item-text">
                                    <ul class="rating">
                                        <?php 
                                        $food_rating = $item['food_rating'];

                                        // Calculate full stars, half stars, and empty stars
                                        $full_stars = floor($food_rating);
                                        $half_star = ($food_rating - $full_stars >= 0.5) ? 1 : 0;
                                        $empty_stars = 5 - ($full_stars + $half_star);
                                        for ($i = 1; $i <= $full_stars; $i++): ?>
                                            <li><i class="fa-solid fa-star"></i></li>
                                        <?php endfor; ?>

                                        <?php if ($half_star): ?>
                                            <li><i class="fa-solid fa-star-half-stroke"></i></li>
                                        <?php endif; ?>

                                        <?php for ($i = 1; $i <= $empty_stars; $i++): ?>
                                            <li><i class="fa-regular fa-star"></i></li>
                                        <?php endfor; ?>
                                    </ul>
                                    <h6 class="food-menu-tab__item-text-title">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php print rrdevs_kses($item['food_menu_name']); ?>
                                        </a>
                                    </h6>
                                    <span class="price">
                                        <?php echo esc_attr($currency); ?><?php print rrdevs_kses($item['food_menu_price']); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

        <?php
	}

}

$widgets_manager->register( new Delishs_Food_Menu_List() );