<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Discount extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_discount';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Discount', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_food',
            [
                'label' => esc_html__('Find Food Lounge', 'delishs-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'food_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Find Food Lounge', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'info_type',
            [
                'label' => esc_html__('Info Type', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'address' => __( 'Address', 'delishs-core' ),
                    'phone' => __( 'Phone', 'delishs-core' ),
                    'email' => __( 'Mail', 'delishs-core' ),
                    'office-time' => __( 'Office Time', 'delishs-core' ),
                ],
                'default' => 'address',
            ]
        );

        $repeater->add_control(
            'contact_info_address', [
                'label' => esc_html__('Address', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'address' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_address_url', [
                'label' => esc_html__('Address URL', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'address' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_phone', [
                'label' => esc_html__('Phone', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'phone' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_phone_url', [
                'label' => esc_html__('Phone URL', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'phone' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_email', [
                'label' => esc_html__('Email', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'email' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_office_time_one', [
                'label' => esc_html__('Office Time One', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'office-time' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_office_time_two', [
                'label' => esc_html__('Office Time Two', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'office-time' ],
                ],
            ]
        );
     
        $this->add_control(
            'food_list',
            [
                'label' => esc_html__('Info List', 'delishs-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'contact_info_address' => __( '25BS, Split Avenue 5896 Frankfurt', 'delishs-core' ),
                        'contact_info_address_url' => __( 'https://maps.app.goo.gl/zL9AWnpXntFCG86U7', 'delishs-core' ),
                        'info_type' => 'address',
                    ],
                    [
                        'contact_info_phone' => __( '+(310) 2591 21563', 'delishs-core' ),
                        'contact_info_phone_url' => '+310259121563',
                        'info_type' => 'phone',
                    ],
                    [
                        'contact_info_email' => 'help-delish@gmail.com',
                        'info_type' => 'email',
                    ],
                    [
                        'contact_info_office_time_one' => __( 'Mon - Thu: 10 AM - 02 AM', 'delishs-core' ),
                        'contact_info_office_time_two' => __( 'Fri - Sun: 10 AM - 02 AM', 'delishs-core' ),
                        'info_type' => 'office-time',
                    ],
                ],
                'title_field' => '{{{ info_type }}}',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_image',
            [
                'label' => esc_html__( 'Offer Image',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'offer_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_contact',
            [
                'label' => esc_html__( 'Contact Us Now',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'contact_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Contact Us Now', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'contact_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Sus nisi platea primis dignissim quam cras netus natoque vel senectus, quis fames praesent sapien...', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'contact_button_text',
            [
                'label' => esc_html__( 'Button Text', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Book A Table', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'contact_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'contact_button_link',
            [
                'label' => esc_html__( 'Button link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'contact_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'contact_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'contact_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_food',
            [
                'label' => esc_html__( 'Find Food Lounge',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_food_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'food_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'food_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text h6' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_title_typography',
                'selector' => '{{WRAPPER}} .discount__item-text h6',
            ]
        );

        $this->add_control(
            '_heading_style_food_title_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text img' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_food_info',
            [
                'label' => esc_html__( 'Info', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'food_info_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text__contact-info li' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'food_info_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text__contact-info li' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .discount__item-text__contact-info li a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'food_info_color_hover',
            [
                'label' => __( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text__contact-info li a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'food_info_typography',
                'selector' => '{{WRAPPER}} .discount__item-text__contact-info li',
            ]
        );

        $this->add_control(
            '_heading_style_food_info_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'food_info_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'food_info_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-food' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_contact',
            [
                'label' => esc_html__( 'Contact Us Now',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_contact_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'contact_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact .discount__item-text h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'contact_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact .discount__item-text h6' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'contact_title_typography',
                'selector' => '{{WRAPPER}} .discount__item-contact .discount__item-text h6',
            ]
        );

        $this->add_control(
            '_heading_style_contact_title_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'contact_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact .discount__item-text img' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_contact_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'contact_description_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact .discount__item-text p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'contact_description_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact .discount__item-text p' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'contact_description_typography',
                'selector' => '{{WRAPPER}} .discount__item-contact .discount__item-text p',
            ]
        );

        $this->add_control(
            '_heading_style_contact_button',
            [
                'label' => esc_html__( 'Button', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->start_controls_tabs( 'tabs_style_contact_button' );
        
        $this->start_controls_tab(
            'contact_button_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'contact_button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button'    => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'contact_button_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'delishs-core' ),
                'name'     => 'contact_button_border',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'contact_button_box_shadow',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'contact_button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'contact_button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'contact_button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'delishs-core' ),
                'name'     => 'contact_button_border_hover',
                'selector' => '{{WRAPPER}} .delishs-el-button:hover',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'contact_button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .delishs-el-button:hover',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();
        
        $this->add_control(
            'contact_button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .delishs-el-button:before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'delishs-core' ),
                'name'     => 'contact_button_typography',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );
        
        $this->add_control(
            'contact_button_padding',
            [
                'label'      => esc_html__( 'Padding', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'contact_button_margin',
            [
                'label'      => esc_html__( 'Margin', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_contact_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'contact_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'contact_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .discount__item-contact' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['offer_image']['url']) ) {
            $offer_image = !empty($settings['offer_image']['id']) ? wp_get_attachment_image_url( $settings['offer_image']['id'], 'full') : $settings['offer_image']['url'];
            $offer_image_alt = get_post_meta($settings["offer_image"]["id"], "_wp_attachment_image_alt", true);
        }

        if ('2' == $settings['contact_button_link_type']) {
            $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['contact_button_page_link']));
            $this->add_render_attribute('delishs-button-arg', 'target', '_self');
            $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button btn-book');
        } else {
            if ( ! empty( $settings['contact_button_link']['url'] ) ) {
                $this->add_link_attributes( 'delishs-button-arg', $settings['contact_button_link'] );
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button btn-book');
            }
        }

		?>

            <section class="delishs-el-section discount overflow-hidden">
                <div class="custom-container">
                    <div class="discount__item discount__item-food theme-bg-primary wow clip-a-z">
                        <div class="discount__item-text">
                            <?php if ( !empty( $settings['food_title'] ) ) : ?>
                                <h6 class="mb-10"><?php print rrdevs_kses($settings['food_title']); ?></h6>
                                <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/discount/dot-shap.svg" class="mb-30 img-fluid" alt="not found">
                            <?php endif; ?>
                            <ul class="discount__item-text__contact-info">
                                <?php foreach ($settings['food_list'] as $item) : ?>
                                    <?php if ( $item['info_type']  == 'address' ): ?>
                                        <li>
                                            <a href="<?php echo esc_url($item['contact_info_address_url' ]); ?>">
                                                <?php echo delishs_kses($item['contact_info_address' ]); ?>
                                            </a>
                                        </li>
                                    <?php elseif ( $item['info_type']  == 'phone' ): ?>
                                        <li>
                                            <a href="tel:<?php echo esc_attr($item['contact_info_phone_url' ]); ?>">
                                                <?php echo delishs_kses($item['contact_info_phone' ]); ?>
                                            </a>
                                        </li>
                                    <?php elseif ( $item['info_type']  == 'email' ): ?>
                                        <li>
                                            <a href="mailto:<?php echo esc_attr($item['contact_info_email' ]); ?>">
                                                <?php echo delishs_kses($item['contact_info_email' ]); ?>
                                            </a>
                                        </li>
                                    <?php elseif ( $item['info_type']  == 'office-time' ): ?>
                                        <li><?php print rrdevs_kses($item['contact_info_office_time_one']); ?></li>
                                        <li><?php print rrdevs_kses($item['contact_info_office_time_two']); ?></li>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>

                    <div class="discount__item wow clip-a-z">
                        <div class="discount__item-media" data-background="<?php print esc_url($offer_image); ?>"></div>
                    </div>

                    <div class="discount__item discount__item-contact wow clip-a-z">
                        <div class="discount__item-text">
                            <?php if ( !empty( $settings['contact_title'] ) ) : ?>
                                <h6 class="mb-10">
                                    <?php print rrdevs_kses($settings['contact_title']); ?>
                                </h6>
                                <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/discount/dot-shap-primary.svg" class="mb-30 img-fluid" alt="not found">
                            <?php endif; ?>
                            <?php if ( !empty( $settings['contact_description'] ) ) : ?>
                                <p class="mb-45 mb-xs-30">
                                    <?php print rrdevs_kses($settings['contact_description']); ?>
                                </p>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['contact_button_text'] ) ) : ?>
                                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                    <?php print rrdevs_kses($settings['contact_button_text']); ?>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </section>

        <?php 
	}
}

$widgets_manager->register( new Delishs_Discount() );