<?php
namespace DelishsCore\Widgets;

use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;
use \Elementor\Widget_Base;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Cta extends Widget_Base {

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'delishs_cta';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return esc_html__( 'CTA', 'delishs-core' );
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'delishs-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return ['delishs_core'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends() {
        return ['delishs-core'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label'   => esc_html__( 'Select Layout', 'delishs-core' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'delishs-core' ),
                    'layout-3' => esc_html__( 'Layout 3', 'delishs-core' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'section_background_image',
            [
                'label'     => esc_html__( 'Background Image', 'delishs-core' ),
                'type'      => Controls_Manager::MEDIA,
                'default'   => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label'       => esc_html__( 'Subtitle', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type'        => Controls_Manager::TEXT,
                'default'     => __( 'CRISPY, EVERY BITE TASTE', 'delishs-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label'       => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type'        => Controls_Manager::TEXT,
                'default'     => __( 'LOOKING FOR POPULAR FAST FOOD', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'title_tag',
            [
                'label'       => esc_html__( 'Title HTML Tag', 'delishs-core' ),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => true,
                'options'     => [
                    'h1' => [
                        'title' => esc_html__( 'H1', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h1',
                    ],
                    'h2' => [
                        'title' => esc_html__( 'H2', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h2',
                    ],
                    'h3' => [
                        'title' => esc_html__( 'H3', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h3',
                    ],
                    'h4' => [
                        'title' => esc_html__( 'H4', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h4',
                    ],
                    'h5' => [
                        'title' => esc_html__( 'H5', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h5',
                    ],
                    'h6' => [
                        'title' => esc_html__( 'H6', 'delishs-core' ),
                        'icon'  => 'eicon-editor-h6',
                    ],
                ],
                'default'     => 'h3',
                'toggle'      => false,
            ]
        );

        $this->end_controls_section();

        // delishs_btn_button_group
        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'delishs_button_text', [
                'label'       => esc_html__( 'Button Text', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type'        => Controls_Manager::TEXT,
                'default'     => esc_html__( 'READ MORE', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'delishs_button_link_type',
            [
                'label'   => esc_html__( 'Service Link Type', 'delishs-core' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );

        $this->add_control(
            'delishs_button_link',
            [
                'label'         => esc_html__( 'Service Link link', 'delishs-core' ),
                'type'          => Controls_Manager::URL,
                'dynamic'       => [
                    'active' => true,
                ],
                'placeholder'   => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default'       => [
                    'url'         => '#',
                    'is_external' => false,
                    'nofollow'    => false,
                ],
                'condition'     => [
                    'delishs_button_link_type' => '1',
                ],
            ]
        );

        $this->add_control(
            'delishs_button_page_link',
            [
                'label'       => esc_html__( 'Select Service Link Page', 'delishs-core' ),
                'type'        => Controls_Manager::SELECT2,
                'label_block' => true,
                'options'     => delishs_get_all_pages(),
                'condition'   => [
                    'delishs_button_link_type' => '2',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_phone',
            [
                'label' => esc_html__( 'Phone',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->add_control(
            'phone_number_label',
            [
                'label' => esc_html__( 'Phone label', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Have Questions?', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'phone_number_label_two',
            [
                'label' => esc_html__( 'Phone Level Two', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Free', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'phone_number',
            [
                'label' => esc_html__( 'Phone', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( '+92 (8800)-9850', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'phone_number_url',
            [
                'label' => esc_html__( 'Phone Link', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( '+9288009850', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background_overlay',
            [
                'label'     => esc_html__( 'Overlay', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fast-food__overlay:after' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_margin',
            [
                'label'      => esc_html__( 'Margin', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_padding',
            [
                'label'      => esc_html__( 'Padding', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .cta-2__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .footer__top-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label'     => esc_html__( 'Title', 'delishs-core' ),
                'type'      => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label'      => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .footer__top-content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label'     => __( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .footer__top-content p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_title_color_hover',
            [
                'label'     => __( 'Color (Hover)', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title a:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title, .footer__top-content p',
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type'  => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label'      => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label'     => __( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label'     => __( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'delishs_button_style',
            [
                'label' => __( 'Button', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__white .btn-wrap .text-one, .rr-btn__white .btn-wrap .text-two' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .cta-2__content .btn-book' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .footer__top-content p a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background_color',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .cta-2__content .btn-book' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_border_color',
            [
                'label'     => esc_html__( 'Border Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__white .fake_hover:before' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'selector' => '{{WRAPPER}} .cta-2__content .btn-book',
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__white .btn-wrap .text-one:hover, .rr-btn__white .btn-wrap .text-two:hover' => 'color: {{VALUE}}!important;',
                    '{{WRAPPER}} .cta-2__content .btn-book:hover' => 'color: {{VALUE}}!important;',
                    '{{WRAPPER}} .footer__top-content p a:hover' => 'color: {{VALUE}}!important;',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .cta-2__content .btn-book:hover' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label'     => esc_html__( 'Border Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn:after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:before' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn .hover-rl:after' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border_hover',
                'selector' => '{{WRAPPER}} .cta-2__content .btn-book:hover',
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .cta-2__content .btn-book' => 'border-radius: {{SIZE}}px;',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'delishs-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .cta-2__content .btn-book',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_phone',
            [
                'label' => esc_html__( 'Phone',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_phone_label_one',
            [
                'label' => esc_html__( 'Label One', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_control(
            'phone_label_one_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .footer__top-contact-btn span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'phone_label_one_typography',
                'selector' => '{{WRAPPER}} .footer__top-contact-btn span',
            ]
        );

        $this->add_control(
            '_heading_style_phone_label_two',
            [
                'label' => esc_html__( 'Label Two', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_control(
            'phone_label_two_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .footer__top-contact-btn h6' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'phone_label_two_typography',
                'selector' => '{{WRAPPER}} .footer__top-contact-btn h6',
            ]
        );

        $this->add_control(
            '_heading_style_phone',
            [
                'label' => esc_html__( 'Phone', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( 'phone_tabs' );
        
        $this->start_controls_tab(
            'phone_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'phone_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .footer__top-contact-btn h6 a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'phone_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'phone_color_hover',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .footer__top-contact-btn h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'phone_typography',
                'selector' => '{{WRAPPER}} .footer__top-contact-btn h6 a',
            ]
        );
        
        $this->end_controls_section();

    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        ?>

        <?php if ( $settings['design_style'] == 'layout-1' ):

            if ( '2' == $settings['delishs_button_link_type'] ) {
                $this->add_render_attribute( 'delishs-button-arg', 'href', get_permalink( $settings['delishs_button_page_link'] ) );
                $this->add_render_attribute( 'delishs-button-arg', 'target', '_self' );
                $this->add_render_attribute( 'delishs-button-arg', 'rel', 'nofollow' );
                $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__white wow clip-a-z' );
            } else {
                if ( !empty( $settings['delishs_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                    $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__white wow clip-a-z' );
                }
            }

            ?>

	            <section class="delishs-el-section fast-food fast-food__overlay section-space" data-background="<?php print esc_url($settings['section_background_image']['url']); ?>">
	                <div class="container">
	                    <div class="row g-24">
	                        <div class="col-12">
	                            <div class="section__title-wrapper fast-food__content text-center">
	                                <?php if ( !empty( $settings['section_subheading'] ) ): ?>
	                                    <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-20 mb-xs-15 wow clip-a-z">
	                                        <?php echo rrdevs_kses( $settings['section_subheading'] ); ?>
	                                    </span>
	                                <?php endif;?>
                                    <?php if ( !empty( $settings['section_title'] ) ): ?>
                                        <h2 class="delishs-el-section-title section__title mb-35 mb-xs-30 title-animation section__title-white section__title-60">
                                            <?php echo rrdevs_kses( $settings['section_title'] ); ?>
                                        </h2>
                                    <?php endif;?>
                                    <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                        <span class="hover-rl"></span>
                                        <span class="fake_hover"></span>
                                        <span class="btn-wrap">
                                            <span class="text-one">
                                                <?php print rrdevs_kses( $settings['delishs_button_text'] );?> <img src="<?php print get_template_directory_uri();?>/assets/imgs/icon/arrow-right-1.svg" alt="not found">
                                            </span>
                                            <span class="text-two">
                                                <?php print rrdevs_kses( $settings['delishs_button_text'] );?> <img src="<?php print get_template_directory_uri();?>/assets/imgs/icon/arrow-right-1.svg" alt="not found">
                                            </span>
                                        </span>
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

        <?php elseif ( $settings['design_style'] == 'layout-2' ):

            if ( '2' == $settings['delishs_button_link_type'] ) {
                $this->add_render_attribute( 'delishs-button-arg', 'href', get_permalink( $settings['delishs_button_page_link'] ) );
                $this->add_render_attribute( 'delishs-button-arg', 'target', '_self' );
                $this->add_render_attribute( 'delishs-button-arg', 'rel', 'nofollow' );
                $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button' );
            } else {
                if ( !empty( $settings['delishs_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                    $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button' );
                }
            }

            ?>

                <div class="delishs-el-section footer__top-wrapper mb-100 mb-xs-70 position-relative z-2" data-background="<?php print esc_url($settings['section_background_image']['url']); ?>">
                    <div class="container">
                        <div class="footer__top">
                            <div class="row g-24 align-items-center">
                                <div class="col-xl-6">
                                    <div class="footer__top-content text-xl-start text-center">
                                        <p class="mb-0">
                                            <?php if ( !empty( $settings['section_title'] ) ): ?>
                                                <?php echo rrdevs_kses( $settings['section_title'] ); ?>
                                            <?php endif;?>
                                            <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                                <?php print rrdevs_kses( $settings['delishs_button_text'] );?>
                                            </a>
                                        </p>
                                    </div>
                                </div>

                                <div class="col-xl-6">
                                    <div class="footer__top-contact-btn">
                                        <div class="footer__top-contact-btn-icon">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/message.svg" alt="not found">
                                        </div>
                                        <div class="footer__top-contact-btn-text">
                                            <span><?php echo rrdevs_kses( $settings['phone_number_label'] ); ?></span>
                                            <h6>
                                                <?php echo rrdevs_kses( $settings['phone_number_label_two'] ); ?> 
                                                <a href="tel:<?php echo esc_attr( $settings['phone_number_url'] ); ?>">
                                                    <?php echo rrdevs_kses( $settings['phone_number'] ); ?>
                                                </a>
                                            </h6>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

        <?php elseif ( $settings['design_style'] == 'layout-3' ):

            if ( '2' == $settings['delishs_button_link_type'] ) {
                $this->add_render_attribute( 'delishs-button-arg', 'href', get_permalink( $settings['delishs_button_page_link'] ) );
                $this->add_render_attribute( 'delishs-button-arg', 'target', '_self' );
                $this->add_render_attribute( 'delishs-button-arg', 'rel', 'nofollow' );
                $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button' );
            } else {
                if ( !empty( $settings['delishs_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                    $this->add_render_attribute( 'delishs-button-arg', 'class', 'delishs-el-button' );
                }
            }

            if ( '2' == $settings['delishs_button_link_type'] ) {
                $this->add_render_attribute( 'delishs-button-arg-two', 'href', get_permalink( $settings['delishs_button_page_link'] ) );
                $this->add_render_attribute( 'delishs-button-arg-two', 'target', '_self' );
                $this->add_render_attribute( 'delishs-button-arg-two', 'rel', 'nofollow' );
                $this->add_render_attribute( 'delishs-button-arg-two', 'class', 'delishs-el-button-two btn-book' );
            } else {
                if ( !empty( $settings['delishs_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg-two', $settings['delishs_button_link'] );
                    $this->add_render_attribute( 'delishs-button-arg-two', 'class', 'delishs-el-button-two btn-book' );
                }
            }

            ?>

            <section class="delishs-el-section cta-2" data-background="<?php print esc_url($settings['section_background_image']['url']); ?>">
                <div class="container">
                    <div class="row g-24">
                        <div class="col-12">
                            <div class="cta-2__content">
                                <div class="cta-2__content-text">
                                    <?php if ( !empty( $settings['section_subheading'] ) ): ?>
                                        <span class="delishs-el-section-subheading text-uppercase"><?php echo rrdevs_kses( $settings['section_subheading'] ); ?></span>
                                    <?php endif;?>
                                    <?php if ( !empty( $settings['section_title'] ) ): ?>
                                        <h3 class="delishs-el-section-title text-uppercase title-animation">
                                            <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                                <?php echo rrdevs_kses( $settings['section_title'] ); ?>
                                            </a>
                                        </h3>
                                    <?php endif;?>
                                </div>
                                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg-two' ); ?>>
                                    <?php print rrdevs_kses($settings['delishs_button_text']); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php endif;?>
        <?php
    }
}

$widgets_manager->register( new Delishs_Cta() );