<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Contact_Info extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_contact_info';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Contact Info', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_contact_info',
            [
                'label' => esc_html__('Find Food Lounge', 'delishs-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'contact_info_google_map_url', 
            [
                'label' => esc_html__( 'Google Map URL', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => __( 'https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d14600.157880481716!2d90.35107202273903!3d23.81719545680701!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x3755c12c1b289e39%3A0x8d3bdfb80d3ebe88!2sRd%20No.%204%2C%20Dhaka%201216!5e0!3m2!1sen!2sbd!4v1701837409513!5m2!1sen!2sbd', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Find Food Lounge', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __( 'Completely recapitalize 24/7 communities via standards compliant metrics whereas.', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $repeater = new Repeater();

        if (delishs_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'contact_info_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-map-marker',
                ]
            );
        } else {
            $repeater->add_control(
                'contact_info_selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'fa fa-map-marker',
                        'library' => 'solid',
                    ],
                ]
            );
        };

        $repeater->add_control(
            'contact_info_title', 
            [
                'label' => esc_html__('Title', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__( 'Our Location', 'delishs-core' ),
            ]
        );

        $repeater->add_control(
            'info_type',
            [
                'label' => esc_html__('Info Type', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'address' => __( 'Address', 'delishs-core' ),
                    'phone' => __( 'Phone', 'delishs-core' ),
                    'office-time' => __( 'Office Time', 'delishs-core' ),
                ],
                'default' => 'address',
            ]
        );

        $repeater->add_control(
            'contact_info_address', [
                'label' => esc_html__('Address', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'address' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_address_url', [
                'label' => esc_html__('Address URL', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'address' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_phone', [
                'label' => esc_html__('Phone', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'phone' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_phone_url', [
                'label' => esc_html__('Phone URL', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'email' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_email', [
                'label' => esc_html__('Email', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'phone' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_office_time_one', [
                'label' => esc_html__('Office Time One', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'office-time' ],
                ],
            ]
        );

        $repeater->add_control(
            'contact_info_office_time_two', [
                'label' => esc_html__('Office Time Two', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'info_type' => [ 'office-time' ],
                ],
            ]
        );
     
        $this->add_control(
            'contact_info_list',
            [
                'label' => esc_html__('Info List', 'delishs-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'contact_info_title' => __( 'Phone Number & Email', 'delishs-core' ),
                        'contact_info_phone' => __( '+(310) 2591 21563', 'delishs-core' ),
                        'contact_info_phone_url' => '+310259121563',
                        'contact_info_email' => 'help-delish@gmail.com',
                        'info_type' => 'phone',
                    ],
                    [
                        'contact_info_title' => __( 'Our Office Address', 'delishs-core' ),
                        'contact_info_address' => __( '258 Dancing Street, Miland Line, HUYI 21563, NewYork', 'delishs-core' ),
                        'contact_info_address_url' => __( 'https://maps.app.goo.gl/1N77c8d8zieLRaot7', 'delishs-core' ),
                        'info_type' => 'address',
                    ],
                    [
                        'contact_info_title' => __( 'Official Work Time', 'delishs-core' ),
                        'contact_info_office_time_one' => __( '7:00am - 6:00pm ( Mon - Fri )', 'delishs-core' ),
                        'contact_info_office_time_two' => __( 'Sat, Sun & <span>Holiday Closed</span>', 'delishs-core' ),
                        'info_type' => 'office-time',
                    ],
                ],
                'title_field' => '{{{ contact_info_title }}}',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            '_section_contact_info_style',
            [
                'label' => __( 'Design Layout', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_title_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper h5' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper h5' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .contact-us__information-wrapper h5',
            ]
        );

        $this->add_control(
            '_heading_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_description_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_description_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .contact-us__information-wrapper p',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_info_list',
            [
                'label' => esc_html__( 'Info List',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_icon',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'info_icon_border_radius',
            [
                'label'      => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px'],
                'selectors'  => [
                    '{{WRAPPER}} .contact-us__information__item .icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'info_icon_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .icon i' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'info_icon_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'info_icon_border',
                'selector' => '{{WRAPPER}} .contact-us__information__item .icon',
            ]
        );

        $this->add_control(
            '_content_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'title_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text span' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .contact-us__information__item .text span',
            ]
        );

        $this->add_control(
            '_heading_info',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Info', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'info_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item:not(:last-child)' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'info_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text a' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .contact-us__information__item .text div' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( 'info_tabs' );
        
        $this->start_controls_tab(
            'info_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'info_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .contact-us__information__item .text div' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'info_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'info_color_hover',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'info_typography',
                'selector' => '{{WRAPPER}} .contact-us__information__item .text a, .contact-us__information__item .text div',
            ]
        );

        $this->add_control(
            'info_highlight_color',
            [
                'label' => esc_html__( 'Highlight Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information__item .text .holiday span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_info_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'info_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'info_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .contact-us__information-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <section class="contact-us overflow-hidden">
                <div class="contact-us__map">
                    <iframe src="<?php print esc_url($settings['contact_info_google_map_url']); ?>" width="600" height="450" style="border:0;" allowfullscreen="" loading="lazy" referrerpolicy="no-referrer-when-downgrade"></iframe>
                </div>

                <div class="container">
                    <div class="contact-us__information-wrapper wow clip-a-z">
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h5 class="text-uppercase mb-10"><?php print rrdevs_kses($settings['section_title']); ?></h5>
                        <?php endif; ?>
                        <?php if ( !empty( $settings['section_description'] ) ) : ?>
                            <p><?php print rrdevs_kses($settings['section_description']); ?></p>
                        <?php endif; ?>
                        <?php foreach ($settings['contact_info_list'] as $item) : ?>
                            <div class="contact-us__information__item">
                                <?php if (!empty($item['contact_info_icon']) || !empty($item['contact_info_selected_icon']['value'])) : ?>
                                    <div class="icon">
                                        <?php delishs_render_icon($item, 'contact_info_icon', 'contact_info_selected_icon'); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ( $item['info_type']  == 'address' ): ?>
                                    <div class="text">
                                        <span><?php print $item['contact_info_title'] ?></span>
                                        <a href="<?php echo esc_url($item['contact_info_address_url' ]); ?>">
                                            <?php echo delishs_kses($item['contact_info_address' ]); ?>
                                        </a>
                                    </div>
                                <?php elseif ( $item['info_type']  == 'phone' ): ?>
                                    <div class="text">
                                        <span><?php print $item['contact_info_title'] ?></span>
                                        <a href="tel:<?php echo esc_attr($item['contact_info_phone_url' ]); ?>">
                                            <?php echo delishs_kses($item['contact_info_phone' ]); ?>
                                        </a>
                                        <a href="mailto:<?php echo esc_attr($item['contact_info_email' ]); ?>">
                                            <?php echo delishs_kses($item['contact_info_email' ]); ?>
                                        </a>
                                    </div>
                                <?php elseif ( $item['info_type']  == 'office-time' ): ?>
                                    <div class="text">
                                        <span><?php print $item['contact_info_title'] ?></span>
                                        <div><?php print rrdevs_kses($item['contact_info_office_time_one']); ?></div>
                                        <div class="holiday"><?php print rrdevs_kses($item['contact_info_office_time_two']); ?></div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php 
	}
}

$widgets_manager->register( new Delishs_Contact_Info() );