<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Chef extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_chef';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Chef', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_chef_list',
            [
                'label' => esc_html__('Chef', 'delishs-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::HIDDEN,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $repeater = new Repeater();                  

        $repeater->add_control(
            'chef_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __( 'Chef Image', 'delishs-core' ),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );                      

        $repeater->add_control(
            'chef_name',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => __( 'Name', 'delishs-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'designation',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'show_label' => true,
                'label' => __( 'Designation', 'delishs-core' ),
                'default' => __( 'Software Tester', 'delishs-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'delishs_link_switcher',
            [
                'label' => esc_html__( 'Show Link', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'delishs-core' ),
                'label_off' => esc_html__( 'No', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'delishs_link_type',
            [
                'label' => esc_html__( 'Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'delishs_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'delishs_link',
            [
                'label' => esc_html__( 'Link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'delishs_link_type' => '1',
                    'delishs_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'delishs_page_link',
            [
                'label' => esc_html__( 'Select Link Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'delishs_link_type' => '2',
                    'delishs_link_switcher' => 'yes',
                ]
            ]
        );

        // REPEATER
        $this->add_control(
            'chef_list',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '<# print(chef_name || "Carousel Item"); #>',
                'default' => [
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'KUMAN TUNMAN', 'delishs-core' ),
                        'designation' => __( 'WAITER', 'delishs-core' ),
                    ],
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'MATTHEW GEORGE', 'delishs-core' ),
                        'designation' => __( 'MANAGER', 'delishs-core' ),
                    ],
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'CHARLES DANIEL', 'delishs-core' ),
                        'designation' => __( 'HEAD CHEF', 'delishs-core' ),
                    ],
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'GEORGE JAMESIEL', 'delishs-core' ),
                        'designation' => __( 'ACCOUNTANT', 'delishs-core' ),
                    ],
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'EDWARD MAY', 'delishs-core' ),
                        'designation' => __( 'SUB-WAITER', 'delishs-core' ),
                    ],
                    [
                        'chef_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'chef_name' => __( 'PATRICK ROBERT', 'delishs-core' ),
                        'designation' => __( 'CO-FOUNDER', 'delishs-core' ),
                    ],
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_column',
            [
                'label' => esc_html__( 'Column',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'delishs-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );
        
        $this->end_controls_section();

		$this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
			'_style_chef_list',
			[
				'label' => __( 'Chef', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        // Name
        $this->add_control(
            '_heading_style_chef_name',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Name', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'chef_name_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .team-3__item-text-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
            'chef_name_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item-text-title' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'chef_name_color_hover',
            [
                'label' => __( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item-text-title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'chef_name_typography',
                'selector' => '{{WRAPPER}} .team-3__item-text-title',
            ]
        );

        // Name
        $this->add_control(
            '_heading_style_chef_designation',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Designation', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

		$this->add_control(
            'chef_designation_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item-text span' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'chef_designation_color_hover',
            [
                'label' => __( 'Color (Hover)', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item-text span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'chef_designation_typography',
                'selector' => '{{WRAPPER}} .team-3__item-text span',
            ]
        );

        $this->add_control(
            '_heading_style_chef_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( 'chef_layout_tabs' );
        
        $this->start_controls_tab(
            'chef_layout_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'chef_name_border',
				'selector' => '{{WRAPPER}} .team-3__item',
			]
		);

		$this->add_control(
            'chef_layout_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'chef_layout_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'chef_name_border_hover',
				'selector' => '{{WRAPPER}} .team-3__item:hover',
			]
		);

        $this->add_control(
            'chef_layout_background_hover',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-3__item:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_responsive_control(
            'chef_name_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .team-3__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>

                <section class="delishs-el-section team-3 section-space">
                    <div class="container">
                        <div class="row g-24 mb-minus-24">
                            <?php foreach ( $settings['chef_list'] as $item ) :

                                if ( !empty($item['chef_image']['url']) ) {
                                    $chef_image_url = !empty($item['chef_image']['id']) ? wp_get_attachment_image_url( $item['chef_image']['id'], 'full') : $item['chef_image']['url'];
                                    $chef_image_alt = get_post_meta($item["chef_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                
                                // Link
                                if ('2' == $item['delishs_link_type']) {
                                    $link = get_permalink($item['delishs_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['delishs_link']['url']) ? $item['delishs_link']['url'] : '';
                                    $target = !empty($item['delishs_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['delishs_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            
                            <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-md-<?php print esc_attr($settings['column_laptop']); ?>  col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?>">
                                <div class="team-3__item">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="team-3__item-media">
                                        <img src="<?php echo esc_url($chef_image_url); ?>" class="img-fluid" alt="<?php echo esc_attr($chef_image_alt); ?>">
                                    </a>
                                    <div class="team-3__item-text text-center">
                                        <h5 class="team-3__item-text-title text-uppercase">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo delishs_kses( $item['chef_name'] ); ?>
                                            </a>
                                        </h5>
                                        <?php if( !empty($item['designation']) ) : ?>
                                            <span><?php echo delishs_kses( $item['designation'] ); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

            <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Delishs_Chef() );