<?php
namespace DelishCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Icons_Manager;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Runok_Button extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'button';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Button', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'runok_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            'runok_btn_button_group',
            [
                'label' => esc_html__('Button', 'delishs-core'),
            ]
        );

        $this->add_control(
            'runok_btn_text',
            [
                'label' => esc_html__('Button Text', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Button Text', 'delishs-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'runok_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'runok_btn_link',
            [
                'label' => esc_html__('Button link', 'delishs-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'runok_btn_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'runok_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => runok_get_all_pages(),
                'condition' => [
                    'runok_btn_link_type' => '2',
                ]
            ]
        );

        $this->add_responsive_control(
            'runok_align',
            [
                'label' => esc_html__('Alignment', 'delishs-core'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'delishs-core'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'delishs-core'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'delishs-core'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};'
                ]
            ]
        );

        $this->add_control(
            'button_icon',
            [
                'label' => esc_html__('Icon', 'delishs-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'fas fa-long-arrow-alt-right',
                    'library' => 'solid',
                ],
            ]
        );

        $this->add_control(
            'button_icon_alignment',
            [
                'label'   => esc_html__( 'Icon Position', 'delishs-core' ),
                'type'    => Controls_Manager::SELECT,
                'default' => 'before-icon',
                'options' => [
                    'before-icon' => esc_html__( 'Before', 'delishs-core' ),
                    'after-icon'  => esc_html__( 'After', 'delishs-core' ),
                ],
                'condition' => [
                    'button_icon[value]!' 	=> '',
                ],
            ]
        );

        $this->add_control(
            'button_icon_spacing',
            [
                'label'     => esc_html__( 'Icon Spacing', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon i' => 'padding-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'padding-left: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'button_icon[value]!' 	=> '',
                ],
            ]
        );

        $this->add_control(
            'button_icon_size',
            [
                'label'     => esc_html__( 'Icon Size', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 14,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'button_icon[value]!' 	=> '',
                ],
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'runok_button_style',
			[
				'label' => __( 'Button', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'delishs-core' ),
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .delishs-el-button:after' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'delishs-core' ),
                'name'     => 'button_border_hover',
                'selector' => '{{WRAPPER}} .delishs-el-button:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .delishs-el-button:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'delishs-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        // Link
        if ('2' == $settings['runok_btn_link_type']) {
            $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['runok_btn_page_link']));
            $this->add_render_attribute('delishs-button-arg', 'target', '_self');
            $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button mt-primary-btn primary-2');
        } else {
            if ( ! empty( $settings['runok_btn_link']['url'] ) ) {
                $this->add_link_attributes( 'delishs-button-arg', $settings['runok_btn_link'] );
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button mt-primary-btn primary-2');
            }
        }

		?>
            <?php if (!empty($settings['runok_btn_text'])) : ?>
                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                    <?php if ( !empty($settings['button_icon']['value']) ) : ?>
                        <span class="<?php echo esc_attr( $settings['button_icon_alignment'] ); ?>">
                            <?php runok_render_icon($settings, 'button_icon'); ?>
                        </span>
                    <?php endif; ?>
                    <?php echo $settings['runok_btn_text']; ?>
                </a>
            <?php endif; ?>
        <?php
	}
}

$widgets_manager->register( new Runok_Button() );