<?php
namespace DelishCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Runok_Brand_Slider extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'brand_slider';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Brand Slider', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'runok_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        // layout Panel
        $this->start_controls_section(
            'runok_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
            ]
        );
        $this->add_control(
            'runok_design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                    'layout-2' => esc_html__('Layout 2', 'delishs-core'),
                    'layout-3' => esc_html__('Layout 3', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'sub_title',
            [
                'label' => esc_html__('Subtitle', 'delishs-core'),
                'description' => runok_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('Our Choose Us', 'delishs-core'),
                'label_block' => true,
                'condition' => [
                    'runok_design_style' => ['layout-2'],
                ]
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => runok_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Project Processing', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'title_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'delishs-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => true,
                'options' => [
                    'h1' => [
                        'title' => esc_html__('H1', 'delishs-core'),
                        'icon' => 'eicon-editor-h1'
                    ],
                    'h2' => [
                        'title' => esc_html__('H2', 'delishs-core'),
                        'icon' => 'eicon-editor-h2'
                    ],
                    'h3' => [
                        'title' => esc_html__('H3', 'delishs-core'),
                        'icon' => 'eicon-editor-h3'
                    ],
                    'h4' => [
                        'title' => esc_html__('H4', 'delishs-core'),
                        'icon' => 'eicon-editor-h4'
                    ],
                    'h5' => [
                        'title' => esc_html__('H5', 'delishs-core'),
                        'icon' => 'eicon-editor-h5'
                    ],
                    'h6' => [
                        'title' => esc_html__('H6', 'delishs-core'),
                        'icon' => 'eicon-editor-h6'
                    ]
                ],
                'default' => 'h3',
                'toggle' => false,
            ]
        );

        $this->add_control(
            'description',
            [
                'label' => esc_html__('Description', 'delishs-core'),
                'description' => runok_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('Softewer being able to crank out videos consistently, localize this them for different regions, and still save resources, time, energy has been a game change', 'delishs-core'),
                'label_block' => true,
                'condition' => [
                    'runok_design_style' => 'layout-2',
                ]
            ]
        );

        $this->add_control(
            'brand_shape_image',
            [
                'label' => esc_html__( 'Choose Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'runok_design_style' => 'layout-2',
                ]
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            'runok_brand_section',
            [
                'label' => __( 'Brand Item', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'runok_brand_image',
            [
                'label' => __( 'Image (Dark)', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'runok_brand_image_light',
            [
                'label' => __( 'Image (Light)', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'runok_link_switcher',
            [
                'label' => esc_html__( 'Show Link', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'delishs-core' ),
                'label_off' => esc_html__( 'No', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'runok_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'runok_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'runok_link',
            [
                'label' => esc_html__( 'Service Link link', 'delishs-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'delishs-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                ],
                'condition' => [
                    'runok_link_type' => '1',
                    'runok_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'runok_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'delishs-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => runok_get_all_pages(),
                'condition' => [
                    'runok_link_type' => '2',
                    'runok_link_switcher' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'runok_brand_slides',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => esc_html__( 'Brand Item', 'delishs-core' ),
                'default' => [
                    [
                        'runok_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'runok_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'runok_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'runok_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'runok_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                ]
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'runok_brand_style',
			[
				'label' => __( 'Brand Item', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
            'content_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .sponsor-wrap .sponsor-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .sponsor-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
		
		$this->add_responsive_control(
            'content_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .sponsor-wrap .sponsor-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .sponsor-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
			'content_background',
			[
				'label' => esc_html__( 'Background', 'delishs-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sponsor-section' => 'background-color: {{VALUE}}',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'runok_design_style' => ['layout-2'],
                ]
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'subheading_popover_toggle',
            [
                'label' => esc_html__( 'Shape', 'delishs-core' ),
                'type' => Controls_Manager::POPOVER_TOGGLE,
                'label_off' => esc_html__( 'Default', 'delishs-core' ),
                'label_on' => esc_html__( 'Custom', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->start_popover();

        $this->add_control(
            '_heading_section_subheading_before',
            [
                'label' => esc_html__( 'Before', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'section_subheading_before',
                'types' => [ 'gradient' ],
                'selector' => '{{WRAPPER}} .section-heading .sub-heading:before',
                'condition' => [
                    'subheading_popover_toggle' => 'yes',
                ],
            ]
        );

        $this->add_control(
            '_heading_section_subheading_after',
            [
                'label' => esc_html__( 'After', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'section_subheading_after',
                'types' => [ 'gradient' ],
                'selector' => '{{WRAPPER}} .section-heading .sub-heading:after',
                'condition' => [
                    'subheading_popover_toggle' => 'yes',
                ],
            ]
        );
        
        $this->end_popover();
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-heading .sub-heading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-heading .sub-heading' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .section-heading .sub-heading',
            ]
        );
        
        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-heading .section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-heading .section-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .section-heading .section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_des',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'section_des_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .brand-content .section-heading p' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_des_typography',
                'selector' => '{{WRAPPER}} .brand-content .section-heading p',
            ]
        );

        $this->add_control(
            '_heading_style_section_img_overlay',
            [
                'label' => esc_html__( 'Image Overlay', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'section_image_overlay_color',
            [
                'label' => __( 'Overlay', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .brand-wrap .brand-img-wrap .brand-img:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            'brand_3_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'runok_design_style' => ['layout-3'],
                ]
            ]
        );
        
        $this->add_responsive_control(
            'brand_3_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .sponsor-text.text-center .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'brand_3_title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sponsor-text.text-center .title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'brand_3_title_typography',
                'selector' => '{{WRAPPER}} .sponsor-text.text-center .title',
            ]
        );
        
        $this->end_controls_section();
		
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( $settings['runok_design_style']  == 'layout-1' ):

		$this->add_render_attribute('title_args', 'class', 'title');

		?>

        <section class="sponsor-section pb-130">
            <div class="container">
                <div class="sponsor-wrap">
                    <div class="sponsor-item item-1">
                        <?php if ( !empty($settings['title' ]) ) :
                            printf( '<%1$s %2$s>%3$s</%1$s>',
                            tag_escape( $settings['title_tag'] ),
                            $this->get_render_attribute_string( '_section_title_args' ),
                            rrdevs_kses( $settings['title' ] )
                            );
                            endif;
                        ?>
                    </div>
                    <?php foreach ($settings['runok_brand_slides'] as $item) : 
                        // Image
                        if ( !empty($item['runok_brand_image']['url']) ) {
                            $runok_brand_image_url = !empty($item['runok_brand_image']['id']) ? wp_get_attachment_image_url( $item['runok_brand_image']['id'], 'full') : $item['runok_brand_image']['url'];
                            $runok_brand_image_alt = get_post_meta($item["runok_brand_image"]["id"], "_wp_attachment_image_alt", true);
                        }

                        // Link
                        if ('2' == $item['runok_link_type']) {
                            $link = get_permalink($item['runok_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = $item['runok_link']['url'];
                            $target = '_blank';
                            $rel = 'nofollow';
                        }
                    ?>
                    <div class="sponsor-item">
                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>"><img src="<?php echo esc_url($runok_brand_image_url); ?>" alt="<?php echo esc_url($runok_brand_image_alt); ?>"></a>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        
        <?php
        elseif ( $settings['runok_design_style']  == 'layout-2' ):

		$this->add_render_attribute('_section_title_args', 'class', 'section-title');

		?>

        <section class="brand-section pb-130">
            <div class="brand-wrap pt-130 pb-130">
                <div class="round-shape"><img src="<?php echo get_template_directory_uri(); ?>/assets/img/shapes/brand-round-shape.png" alt="shape"></div>
                <div class="brand-img-wrap">
                    <div class="brand-img">
                        <?php $brand_shape_image = $settings['brand_shape_image']; 
                            if(!empty($brand_shape_image['url'])):
                        ?>
                            <img src="<?php echo esc_url($brand_shape_image['url']); ?>" alt="<?php echo esc_attr($brand_shape_image['alt']); ?>">
                        <?php endif; ?>
                    </div>
                </div>
                <div class="container">
                    <div class="row gy-lg-0 gy-0">
                        <div class="col-lg-7">
                            <div class="brand-content">
                                <div class="section-heading">
                                <?php if(!empty($settings['sub_title'])): ?>
                                    <h4 class="sub-heading after-none" data-text-animation="fade-in" data-duration="1.5"><?php echo rrdevs_kses($settings['sub_title']); ?></h4>
                                <?php endif; ?>
                                <?php if ( !empty($settings['title' ]) ) :
                                    printf( '<%1$s %2$s>%3$s</%1$s>',
                                    tag_escape( $settings['title_tag'] ),
                                    $this->get_render_attribute_string( '_section_title_args' ),
                                    rrdevs_kses( $settings['title' ] )
                                    );
                                    endif;
                                ?>
                                <?php if(!empty($settings['description'])): ?>
                                    <p><?php echo rrdevs_kses($settings['description']); ?></p>
                                <?php endif; ?>
                                </div>
                                <ul class="brand-list">
                                <?php foreach ($settings['runok_brand_slides'] as $item) : 
                                    // Image
                                    if ( !empty($item['runok_brand_image']['url']) ) {
                                        $runok_brand_image_url = !empty($item['runok_brand_image']['id']) ? wp_get_attachment_image_url( $item['runok_brand_image']['id'], 'full') : $item['runok_brand_image']['url'];
                                        $runok_brand_image_alt = get_post_meta($item["runok_brand_image"]["id"], "_wp_attachment_image_alt", true);
                                    }
                                    // Image
                                    if ( !empty($item['runok_brand_image_light']['url']) ) {
                                        $runok_brand_image_light_url = !empty($item['runok_brand_image_light']['id']) ? wp_get_attachment_image_url( $item['runok_brand_image_light']['id'], 'full') : $item['runok_brand_image_light']['url'];
                                        $runok_brand_image_light_alt = get_post_meta($item["runok_brand_image_light"]["id"], "_wp_attachment_image_alt", true);
                                    }

                                    // Link
                                    if ('2' == $item['runok_link_type']) {
                                        $link = get_permalink($item['runok_page_link']);
                                        $target = '_self';
                                        $rel = 'nofollow';
                                    } else {
                                        $link = $item['runok_link']['url'];
                                        $target = '_blank';
                                        $rel = 'nofollow';
                                    }
                                ?>
                                    <li>
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <img class="dark-img" src="<?php echo esc_url($runok_brand_image_url); ?>" alt="<?php echo esc_url($runok_brand_image_alt); ?>">
                                            <img class="light-img" src="<?php echo esc_url($runok_brand_image_light_url); ?>" alt="<?php echo esc_url($runok_brand_image_light_alt); ?>">
                                        </a>
                                    </li>
                                <?php endforeach; ?> 
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>
        <!-- ./ brand-section -->

        <?php
        elseif ( $settings['runok_design_style']  == 'layout-3' ):

		$this->add_render_attribute('_section_title_args', 'class', 'title mb-50');

		?>

        <section id="sponsor" class="sponsor-section p-relative z-1 pb-130">
            <div class="container">
                <div class="sponsor-text text-center">
                    <?php if ( !empty($settings['title' ]) ) :
                        printf( '<%1$s %2$s>%3$s</%1$s>',
                        tag_escape( $settings['title_tag'] ),
                        $this->get_render_attribute_string( '_section_title_args' ),
                        rrdevs_kses( $settings['title' ] )
                        );
                        endif;
                    ?>
                </div>
                <div class="sponsor-carousel-2 swiper">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['runok_brand_slides'] as $item) : 
                            // Image
                            if ( !empty($item['runok_brand_image']['url']) ) {
                                $runok_brand_image_url = !empty($item['runok_brand_image']['id']) ? wp_get_attachment_image_url( $item['runok_brand_image']['id'], 'full') : $item['runok_brand_image']['url'];
                                $runok_brand_image_alt = get_post_meta($item["runok_brand_image"]["id"], "_wp_attachment_image_alt", true);
                            }

                            // Link
                            if ('2' == $item['runok_link_type']) {
                                $link = get_permalink($item['runok_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = $item['runok_link']['url'];
                                $target = '_blank';
                                $rel = 'nofollow';
                            }
                        ?>
                            <div class="swiper-slide">
                                <div class="sponsor-item text-center">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>"><img src="<?php echo esc_url($runok_brand_image_url); ?>" alt="<?php echo esc_url($runok_brand_image_alt); ?>"></a>
                                </div>
                            </div>
                        <?php endforeach; ?> 
                    </div>
                </div>
            </div>
        </section>
        <!-- ./ sponsor-section -->

		<?php
        endif;
	}
}

$widgets_manager->register( new Runok_Brand_Slider() );