<?php
namespace DelishsCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Border;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delishs Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_Blog_Post extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_blog_post';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Blog Post', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Style', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'delishs-core' ),
                    'layout-2'  => esc_html__( 'Layout 2', 'delishs-core' ),
                    'layout-3'  => esc_html__( 'Layout 3', 'delishs-core' ),
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Our Blog Posts', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'latest news insights', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title_shape',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            'post_query',
            [
                'label' => esc_html__('Blog Query', 'delishs-core'),
            ]
        );

        $post_type = 'post';
        $taxonomy = 'category';

        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Posts Per Page', 'delishs-core'),
                'description' => esc_html__('Leave blank or enter -1 for all.', 'delishs-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '3',
            ]
        );

        $this->add_control(
            'category',
            [
                'label' => esc_html__('Include Categories', 'delishs-core'),
                'description' => esc_html__('Select a category to include or leave blank for all.', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => delishs_get_categories($taxonomy),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'exclude_category',
            [
                'label' => esc_html__('Exclude Categories', 'delishs-core'),
                'description' => esc_html__('Select a category to exclude', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => delishs_get_categories($taxonomy),
                'label_block' => true
            ]
        );

        $this->add_control(
            'post__not_in',
            [
                'label' => esc_html__('Exclude Item', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'options' => delishs_get_all_types_post($post_type),
                'multiple' => true,
                'label_block' => true
            ]
        );

        $this->add_control(
            'offset',
            [
                'label' => esc_html__('Offset', 'delishs-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '0',
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order By', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => array(
			        'ID' => 'Post ID',
			        'author' => 'Post Author',
			        'title' => 'Title',
			        'date' => 'Date',
			        'modified' => 'Last Modified Date',
			        'parent' => 'Parent Id',
			        'rand' => 'Random',
			        'comment_count' => 'Comment Count',
			        'menu_order' => 'Menu Order',
			    ),
                'default' => 'date',
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'asc' 	=> esc_html__( 'Ascending', 'delishs-core' ),
                    'desc' 	=> esc_html__( 'Descending', 'delishs-core' )
                ],
                'default' => 'desc',

            ]
        );
        $this->add_control(
            'ignore_sticky_posts',
            [
                'label' => esc_html__( 'Ignore Sticky Posts', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'delishs-core' ),
                'label_off' => esc_html__( 'No', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'blog_title_word',
            [
                'label' => esc_html__('Title Word Count', 'delishs-core'),
                'description' => esc_html__('Set how many word you want to displa!', 'delishs-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '6',
            ]
        );

        $this->add_control(
            'post_content',
            [
                'label' => __('Content', 'morat-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'morat-core'),
                'label_off' => __('Hide', 'morat-core'),
                'return_value' => 'yes',
                'default' => '',
            ]
        );

        $this->add_control(
            'post_content_limit',
            [
                'label' => __('Content Limit', 'morat-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => '14',
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'post_content' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'btn_button_show',
            [
                'label' => esc_html__( 'Show Button', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button Text', 'delishs-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Read More', 'delishs-core'),
                'label_block' => true,
                'condition' => [
                    'btn_button_show' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'delishs_button_text',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Button Text', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'delishs_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'delishs_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'delishs_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'delishs_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'delishs_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_layout',
            [
                'label' => esc_html__( 'Layout', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_layout_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-title-layout' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // style control
		$this->start_controls_section(
			'post_query_style',
			[
				'label' => __( 'Blog Query', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_control(
            '_heading_post_meta',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Post Meta', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'post_meta_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .blog__item-meta-tags' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .blog-2__item-author' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .blog-3__item-meta' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( 'post_meta_tabs' );
        
        $this->start_controls_tab(
            'post_meta_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'post_meta_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-meta-tags a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog__item-date' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item-author li span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item-meta li a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'post_meta_background',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-meta-tags a' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .blog__item-date' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .blog__item-date::before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'post_meta_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'post_meta_color_hover',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-meta-tags a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item-meta li a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'post_meta_background_hover',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-meta-tags a:hover' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'post_meta_typography',
                'selector' => '{{WRAPPER}} .blog__item-meta-tags a, .blog-2__item-author li span, .blog-3__item-meta li a',
            ]
        );
        $this->add_control(
            '_heading_post_comment',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Comment', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->start_controls_tabs( 'post_comment_tabs' );
        
        $this->start_controls_tab(
            'post_comment_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            'post_comment_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__bottom-text a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item-author li a' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'post_comment_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            'post_comment_color_hover',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__bottom-text a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item-author li a:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'post_comment_typography',
                'selector' => '{{WRAPPER}} .blog__item-content__bottom-text a, .blog-2__item-author li a',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__top h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .blog-2__item blog-2__item-text h5' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .blog-3__item h6' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( '_title_style_tabs' );
        
        $this->start_controls_tab(
            'title_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__top h6' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item blog-2__item-text h5' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item h6' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'title_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__top h6 a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item blog-2__item-text h5 a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .blog__item-content__top h6, .blog-2__item blog-2__item-text h5, .blog-3__item h6',
            ]
        );

        $this->add_control(
            '_heading_style_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .blog-3__item__content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );
        
        $this->add_control(
            'description_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog-3__item__content p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .blog-3__item__content p',
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            '_content_link',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Link', 'delishs-core' ),
                'separator' => 'before'
            ]
        );

        $this->start_controls_tabs( '_button_style_tabs' );
        
        $this->start_controls_tab(
            'button_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'button_color',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__bottom-readmore' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item .read-more' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-one' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'button_color_hover',
            [
                'label' => __( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__bottom-readmore:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item .read-more:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-two' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'selector' => '{{WRAPPER}} .blog__item-content__bottom-readmore, .blog-3__item .read-more, .rr-btn__red .btn-wrap .text-one, .rr-btn__red .btn-wrap .text-two',
            ]
        );

        // Link
        $this->add_control(
            '_blog_layout_style',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Layout', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_responsive_control(
            'blog_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .blog-3__item__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            'blog_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content__top' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item-meta' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'blog_layout_background_color',
            [
                'label' => __( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog__item-content' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .blog-2__item' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .blog-3__item__content' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();
        
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } else if (get_query_var('page')) {
            $paged = get_query_var('page');
        } else {
            $paged = 1;
        }

        // include_categories
        $category_list = '';
        if (!empty($settings['category'])) {
            $category_list = implode(", ", $settings['category']);
        }
        $category_list_value = explode(" ", $category_list);

        // exclude_categories
        $exclude_categories = '';
        if(!empty($settings['exclude_category'])){
            $exclude_categories = implode(", ", $settings['exclude_category']);
        }
        $exclude_category_list_value = explode(" ", $exclude_categories);

        $post__not_in = '';
        if (!empty($settings['post__not_in'])) {
            $post__not_in = $settings['post__not_in'];
            $args['post__not_in'] = $post__not_in;
        }
        $posts_per_page = (!empty($settings['posts_per_page'])) ? $settings['posts_per_page'] : '-1';
        $orderby = (!empty($settings['orderby'])) ? $settings['orderby'] : 'post_date';
        $order = (!empty($settings['order'])) ? $settings['order'] : 'desc';
        $offset_value = (!empty($settings['offset'])) ? $settings['offset'] : '0';
        $ignore_sticky_posts = (! empty( $settings['ignore_sticky_posts'] ) && 'yes' == $settings['ignore_sticky_posts']) ? true : false ;


        // number
        $off = (!empty($offset_value)) ? $offset_value : 0;
        $offset = $off + (($paged - 1) * $posts_per_page);
        $p_ids = array();

        // build up the array
        if (!empty($settings['post__not_in'])) {
            foreach ($settings['post__not_in'] as $p_idsn) {
                $p_ids[] = $p_idsn;
            }
        }

        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $posts_per_page,
            'orderby' => $orderby,
            'order' => $order,
            'offset' => $offset,
            'paged' => $paged,
            'post__not_in' => $p_ids,
            'ignore_sticky_posts' => $ignore_sticky_posts
        );

        // exclude_categories
        if ( !empty($settings['exclude_category'])) {

            // Exclude the correct cats from tax_query
            $args['tax_query'] = array(
                array(
                    'taxonomy'	=> 'category',
                    'field'	 	=> 'slug',
                    'terms'		=> $exclude_category_list_value,
                    'operator'	=> 'NOT IN'
                )
            );

            // Include the correct cats in tax_query
            if ( !empty($settings['category'])) {
                $args['tax_query']['relation'] = 'AND';
                $args['tax_query'][] = array(
                    'taxonomy'	=> 'category',
                    'field'		=> 'slug',
                    'terms'		=> $category_list_value,
                    'operator'	=> 'IN'
                );
            }

        } else {
            // Include the cats from $cat_slugs in tax_query
            if (!empty($settings['category'])) {
                $args['tax_query'][] = [
                    'taxonomy' => 'category',
                    'field' => 'slug',
                    'terms' => $category_list_value,
                ];
            }
        }

        $filter_list = $settings['category'];

        // The Query
        $query = new \WP_Query($args);

        ?>

            <?php if ( $settings['design_style']  == 'layout-1' ): 

                    if ('2' == $settings['delishs_button_link_type']) {
                        $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['delishs_button_page_link']));
                        $this->add_render_attribute('delishs-button-arg', 'target', '_self');
                        $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
                        $this->add_render_attribute('delishs-button-arg', 'class', 'rr-btn wow clip-a-z mb-15');
                    } else {
                        if ( ! empty( $settings['delishs_button_link']['url'] ) ) {
                            $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                            $this->add_render_attribute('delishs-button-arg', 'class', 'rr-btn wow clip-a-z mb-15');
                        }
                    }
                ?>

                <section class="blog section-space delishs-el-section">
                    <div class="container">
                        <div class="row delishs-title-layout g-24 mb-70 mb-sm-50 mb-xs-40 align-items-lg-end align-items-center">
                            <div class="col-xl-6">
                                <div class="section__title-wrapper text-center text-xl-start">
                                    <?php if ( !empty ( $settings['section_subheading'] ) ) : ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-start mb-10 mb-xs-5 wow clip-a-z">
                                            <?php print rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty ( $settings['section_title'] ) ) : ?>
                                        <h2 class="delishs-el-section-title section__title mb-0 title-animation">
                                            <?php print rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="col-xl-6">
                                <div class="d-flex justify-content-xl-end mt-xs-10 mt-sm-10 mt-md-10 mt-lg-10 justify-content-center">
                                    <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                        <span class="hover-rl"></span>
                                        <span class="fake_hover"></span>
                                        <span class="btn-wrap">
                                            <span class="text-one">
                                                <?php print rrdevs_kses($settings['delishs_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-1.svg" alt="not found">
                                            </span>
                                            <span class="text-two">
                                                <?php print rrdevs_kses($settings['delishs_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-1.svg" alt="not found">
                                            </span>
                                        </span>
                                    </a>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24 mb-minus-24">
                            <?php if ($query->have_posts()) : ?>
                                <?php while ($query->have_posts()) : 
                                    $query->the_post();
                                    global $post;
                                    $categories = get_the_category($post->ID);
                                ?>
                                <div class="col-xl-4 col-md-6">
                                    <div class="blog__item mb-24">
                                        <div class="blog__item-media">
                                            <a href="<?php the_permalink(); ?>">
                                                <?php the_post_thumbnail('full');?>
                                            </a>
                                            <div class="blog__item-date">
                                                <?php print get_the_date('d'); ?> <span><?php print get_the_date('M'); ?></span>
                                            </div>
                                        </div>

                                        <div class="blog__item-content">
                                            <div class="blog__item-content__top">
                                                <div class="blog__item-meta-tags d-flex flex-wrap">
                                                    <?php if ( !empty( $categories[0]->name ) ): ?>
                                                        <a href="<?php print esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                            <?php echo esc_html($categories[0]->name); ?>
                                                        </a>
                                                    <?php endif;?>
                                                    <?php if ( !empty( $categories[1]->name ) ): ?>
                                                        <a href="<?php print esc_url(get_category_link($categories[1]->term_id)); ?>">
                                                            <?php echo esc_html($categories[1]->name); ?>
                                                        </a>
                                                    <?php endif;?>
                                                </div>
                                                <h6 class="mt-15">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                    </a>
                                                </h6>
                                            </div>
                                            <div class="blog__item-content__bottom">
                                                <div class="blog__item-content__bottom-text">
                                                    <a href="<?php comments_link();?>">
                                                        <i class="fa-solid fa-messages"></i>
                                                        <span><?php comments_number(); ?></span>
                                                    </a>
                                                </div>
                                                <a href="<?php the_permalink(); ?>" class="blog__item-content__bottom-readmore">
                                                    <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <path d="M6.5 1C7.7 1 14 1 17 1M17 1V10.5M17 1L1 17" stroke="#74787C" stroke-width="2"/>
                                                    </svg>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endwhile; wp_reset_query(); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <section class="blog-2 section-space delishs-el-section">
                    <div class="container">
                        <div class="row g-24">
                            <div class="col-12">
                                <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                    <?php if ( !empty ( $settings['section_subheading'] ) ) : ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                            <?php print rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty ( $settings['section_title'] ) ) : ?>
                                        <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                            <?php print rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                        <div class="dot-shpae wow clip-a-z">
                                            <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24">
                            <div class="col-12">
                                <div class="blog-2__item-wrapper section-bg-2">
                                    <?php 
                                        $post_counter = 1;
                                        if ($query->have_posts()) : ?>
                                        <?php while ($query->have_posts()) : 
                                            $query->the_post();
                                            global $post;
                                            $categories = get_the_category($post->ID);
                                        
                                        if ($post_counter % 2 != 0) : ?>
                                            <?php if (has_post_thumbnail()) : ?>
                                                <div class="blog-2__item">
                                                    <div class="blog-2__item-date">
                                                        <?php print get_the_date('d'); ?><span><?php print get_the_date('M'); ?></span>
                                                    </div>
                                                    <a href="<?php the_permalink(); ?>" class="blog-2__item-media wow clip-a-z">
                                                        <?php the_post_thumbnail('full');?>
                                                    </a>
                                                </div>
                                            <?php endif; ?>

                                            <div class="blog-2__item blog-2__item-text">
                                                <ul class="blog-2__item-author mb-20">
                                                    <li>
                                                        <?php print get_avatar( get_the_author_meta( 'user_email' ), 67, '', '', [ 'class' => 'img-responsive' ] );?> 
                                                        <span><?php print get_the_author(); ?></span>
                                                    </li>
                                                    <li>
                                                        <a href="<?php comments_link();?>">
                                                            <i class="fa-solid fa-messages"></i>
                                                            <span><?php comments_number(); ?></span>
                                                        </a>
                                                    </li>
                                                </ul>

                                                <h5 class="mb-70 mt-xl-30 mb-lg-30 mb-md-30 mb-sm-25 mb-xs-20 text-uppercase">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                    </a>
                                                </h5>

                                                <a href="<?php the_permalink(); ?>" class="rr-btn rr-btn__red">
                                                    <span class="hover-rl"></span>
                                                    <span class="fake_hover"></span>
                                                    <span class="btn-wrap">
                                                        <span class="text-one">
                                                            <?php print rrdevs_kses($settings['btn_text']); ?>
                                                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                        </span>
                                                        <span class="text-two">
                                                            <?php print rrdevs_kses($settings['btn_text']); ?>
                                                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                        </span>
                                                    </span>
                                                </a>
                                            </div>
                                        <?php else : ?>
                                            <div class="blog-2__item blog-2__item-text">
                                                <ul class="blog-2__item-author mb-20">
                                                    <li>
                                                        <?php print get_avatar( get_the_author_meta( 'user_email' ), 67, '', '', [ 'class' => 'img-responsive' ] );?> 
                                                        <span><?php print get_the_author(); ?></span>
                                                    </li>
                                                    <li>
                                                        <a href="<?php comments_link();?>">
                                                            <i class="fa-solid fa-messages"></i>
                                                            <span><?php comments_number(); ?></span>
                                                        </a>
                                                    </li>
                                                </ul>

                                                <h5 class="mb-70 mt-xl-30 mb-lg-30 mb-md-30 mb-sm-25 mb-xs-20 text-uppercase">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                    </a>
                                                </h5>

                                                <a href="<?php the_permalink(); ?>" class="rr-btn rr-btn__red">
                                                    <span class="hover-rl"></span>
                                                    <span class="fake_hover"></span>
                                                    <span class="btn-wrap">
                                                        <span class="text-one">
                                                            <?php print rrdevs_kses($settings['btn_text']); ?>
                                                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                        </span>
                                                        <span class="text-two">
                                                            <?php print rrdevs_kses($settings['btn_text']); ?>
                                                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                                        </span>
                                                    </span>
                                                </a>
                                            </div>

                                            <?php if (has_post_thumbnail()) : ?>
                                                <div class="blog-2__item">
                                                    <div class="blog-2__item-date">
                                                        <?php print get_the_date('d'); ?><span><?php print get_the_date('M'); ?></span>
                                                    </div>
                                                    <a href="<?php the_permalink(); ?>" class="blog-2__item-media wow clip-a-z">
                                                        <?php the_post_thumbnail('full');?>
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    <?php
                                        $post_counter++;
                                        endwhile;
                                        wp_reset_query();
                                    ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-3' ): ?>

                <section class="blog-3 section-space delishs-el-section">
                    <div class="container">
                        <div class="row g-24">
                            <div class="col-12">
                                <div class="section__title-wrapper text-center mb-70 mb-sm-50 mb-xs-40">
                                    <?php if ( !empty ( $settings['section_subheading'] ) ) : ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-10 mb-xs-5 wow clip-a-z">
                                            <?php print rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty ( $settings['section_title'] ) ) : ?>
                                        <h2 class="delishs-el-section-title section__title mb-10 mb-xs-5 title-animation">
                                            <?php print rrdevs_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title_shape'] ) ) : ?>
                                        <div class="dot-shpae wow clip-a-z">
                                            <img class="img-fluid" src="<?php print get_template_directory_uri(); ?>/assets/imgs/section-title/dot-shape.svg" alt="not found">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="row g-24 mb-minus-24">
                            <?php if ($query->have_posts()) : 
                                $post_count = 0;
                                ?>
                                <?php while ($query->have_posts()) : 
                                    $query->the_post();
                                    global $post;
                                    $categories = get_the_category($post->ID);
                                    $post_content_limit = $settings['post_content_limit'];
                                    if ($post_count % 3 == 1) {
                                        $col_class = 'col-xl-6';
                                    } else {
                                        $col_class = 'col-xl-3 col-sm-6';
                                    }
                                    $post_count++;
                                    ?>
                                    <div class="<?php echo esc_attr( $col_class ); ?> blog-3__item-col">
                                        <div class="blog-3__item mb-24">
                                            <?php if (has_post_thumbnail()) : ?>
                                                <a href="<?php the_permalink(); ?>" class="blog-3__item-media">
                                                    <?php the_post_thumbnail('full');?>
                                                </a>
                                            <?php endif; ?>
                                            <div class="blog-3__item__content">
                                                <ul class="blog-3__item-meta d-flex flex-wrap">
                                                    <li>
                                                        <a href="<?php the_permalink(); ?>">
                                                            <i class="fa-regular fa-circle-user"></i>
                                                            <span><?php print esc_html( 'By:', 'delishs' ); ?> <?php print get_the_author(); ?></span>
                                                        </a>
                                                    </li>
                                                    <li>
                                                        <a href="<?php the_permalink(); ?>">
                                                            <i class="fa-light fa-tags"></i>
                                                            <span><?php echo esc_html($categories[0]->name); ?></span>
                                                        </a>
                                                    </li>
                                                </ul>
                                                <h6 class="mb-20">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                    </a>
                                                </h6>
                                                <?php if ($col_class == 'col-xl-6') : ?>
                                                    <p class="mb-15">
                                                        <?php echo wp_trim_words(get_the_content(), $post_content_limit, ''); // Display additional content ?>
                                                    </p>
                                                <?php endif; ?>
                                                <a class="read-more" href="<?php the_permalink(); ?>">
                                                    <?php print rrdevs_kses($settings['btn_text']); ?> <i class="fa-light fa-arrow-right"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endwhile; wp_reset_query(); ?>
                            <?php endif; ?>
                        </div>

                    </div>
                </section>

            <?php endif; ?>
            
       <?php
	}

}

$widgets_manager->register( new Delishs_Blog_Post() );