<?php
namespace DelishsCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Background;
use \Elementor\Repeater;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Delish Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Delishs_About extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'delishs_about';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'About', 'delishs-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'delishs-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'delishs_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'delishs-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'delishs-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'delishs-core'),
                    'layout-2' => esc_html__('Layout 2', 'delishs-core'),
                    'layout-3' => esc_html__('Layout 3', 'delishs-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__('Subtitle', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'About Our Company', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Welcome To Our Luxury Restaurant', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__('Description', 'delishs-core'),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('The scallops were perfectly cooked, tender, and flavorful, paired beautifully with a creamy risotto and seasonal vegetables. The presentation was artful, showcasing the chefs attention to detail.', 'delishs-core'),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_image',
            [
                'label' => esc_html__('Image', 'delishs-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'delishs_image',
            [
                'label' => esc_html__( 'Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'delishs_image_two',
            [
                'label' => esc_html__( 'Image Two', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_phone',
            [
                'label' => esc_html__( 'Phone',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            'phone_icon_type',
            [
                'label' => esc_html__( 'Image Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'icon' => esc_html__( 'Icon', 'delishs-core' ),
                    'image' => esc_html__( 'Image', 'delishs-core' ),
                ],
            ]
        );
        
        $this->add_control(
            'phone_image_icon',
            [
                'label' => esc_html__( 'Upload Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'phone_icon_type' => 'image',
                ],
            ]
        );
        
        $this->add_control(
            'phone_icon',
            [
                'label' => esc_html__( 'Icon', 'delishs-core' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'solid',
                ],
                'condition' => [
                    'phone_icon_type' => 'icon',
                ],
            ]
        );

        $this->add_control(
            'phone_label',
            [
                'label' => esc_html__( 'Label', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'HOTLINE 24/7', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'phone_number',
            [
                'label' => esc_html__( 'Number', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( '+256 3254-2568', 'delishs-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'phone_number_link',
            [
                'label' => esc_html__( 'Number Link', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( '+25632542568', 'delishs-core' ),
                'label_block' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_feature',
            [
                'label' => esc_html__('Feature', 'delishs-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'feature_icon_type',
            [
                'label' => esc_html__( 'Image Type', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'icon' => esc_html__( 'Icon', 'delishs-core' ),
                    'image' => esc_html__( 'Image', 'delishs-core' ),
                ],
            ]
        );
        
        $repeater->add_control(
            'feature_image_icon',
            [
                'label' => esc_html__( 'Upload Image', 'delishs-core' ),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'feature_icon_type' => 'image',
                ],
            ]
        );
        
        $repeater->add_control(
            'feature_icon',
            [
                'label' => esc_html__('Icon', 'delishs-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'fas fa-circle-check',
                    'library' => 'brand',
                ],
                'condition' => [
                    'feature_icon_type' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'feature_text',
            [
                'label' => esc_html__( 'Text', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'feature_description',
            [
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'description' => delishs_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'feature_list',
            [
                'label' => esc_html__('Feature - List One', 'delishs-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'feature_text' => __( 'Quality Food', 'delishs-core' ),
                        'feature_description' => __( 'The wise man therefore always holds Indies matters this principle information', 'delishs-core' ),
                    ],
                    [
                        'feature_text' => __( 'Supply Chain Maintain', 'delishs-core' ),
                        'feature_description' => __( 'The wise man therefore always holds Indies matters this principle information', 'delishs-core' ),
                    ],
                ],
                'title_field' => '{{{ feature_text }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__('Button', 'delishs-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'delishs_button_show',
            [
                'label' => esc_html__( 'Show Button', 'delishs-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'delishs-core' ),
                'label_off' => esc_html__( 'Hide', 'delishs-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'delishs_button_text',
            [
                'label' => esc_html__( 'Button Text', 'delishs-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Get Started Now', 'delishs-core' ),
                'label_block' => true,
                'condition' => [
                    'delishs_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'delishs_button_link_type',
            [
                'label' => esc_html__('Button Link Type', 'delishs-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'delishs_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'delishs_button_link',
            [
                'label' => esc_html__('Button link', 'delishs-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'delishs-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'delishs_button_link_type' => '1',
                    'delishs_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'delishs-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => delishs_get_all_pages(),
                'condition' => [
                    'delishs_button_link_type' => '2',
                    'delishs_button_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_design_layout_inner',
            [
                'label' => esc_html__( 'Inner', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            'design_layout_inner_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .reservation-wrapper' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_image',
            [
                'label' => esc_html__( 'Image', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            'design_layout_image_border',
            [
                'label' => esc_html__( 'Border', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-2__media:after' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .reservation__media-small-thumb::after' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_section_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'delishs-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-subheading',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_padding',
            [
                'label' => esc_html__( 'Padding', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_subheading_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'delishs-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-subheading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'delishs-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-title',
            ]
        );

        $this->add_control(
            '_style_heading_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Description', 'delishs-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'section_description_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .section__title-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_description_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-section-description' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .delishs-el-section-description',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_phone',
            [
                'label' => esc_html__( 'Phone',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            '_heading_style_phone_icon',
            [
                'label' => esc_html__( 'Icon', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'phone_icon_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn-icon' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'phone_icon_background',
            [
                'label' => esc_html__( 'Background', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn-icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_phone_label',
            [
                'label' => esc_html__( 'Label', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'phone_label_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn span' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'phone_label_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'phone_label_typography',
                'selector' => '{{WRAPPER}} .about-our-company__content-btn span',
            ]
        );

        $this->add_control(
            '_heading_style_phone_number',
            [
                'label' => esc_html__( 'Number', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( 'phone_number_tabs' );
        
        $this->start_controls_tab(
            'phone_number_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );
        
        $this->add_control(
            'phone_number_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn h6 a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'phone_number_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'phone_number_color_hover',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-our-company__content-btn h6 a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'phone_number_typography',
                'selector' => '{{WRAPPER}} .about-our-company__content-btn h6',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_feature',
            [
                'label' => esc_html__( 'Feature', 'delishs-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            '_style_heading_feature_icon',
            [
                'label' => esc_html__( 'Icon', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'feature_icon_size',
            [
                'label' => esc_html__( 'Icon Size', 'delishs-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .about-2__content-item__media i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'feature_icon_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-2__content-item__media i' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'feature_icon_border_color',
            [
                'label' => esc_html__( 'Border', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-2__content-item__text::before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .about-2__content-item__text::after' => 'border-color: transparent {{VALUE}} transparent transparent',
                ],
            ]
        );

        $this->add_control(
            '_style_heading_feature',
            [
                'label' => esc_html__( 'Feature', 'delishs-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'feature_text_color',
            [
                'label' => esc_html__( 'Color', 'delishs-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .about-2__content-item__text h6' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'feature_text_typography',
                'selector' => '{{WRAPPER}} .about-2__content-item__text h6',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_button',
			[
				'label' => __( 'Button', 'delishs-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-3' ],
                ],
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-one' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .rr-btn-solid .btn-wrap .text-one' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'delishs-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rr-btn__red .btn-wrap .text-two' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .rr-btn-solid .btn-wrap .text-two' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'delishs-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button:hover' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'delishs-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .delishs-el-button' => 'border-radius: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .delishs-el-button:before' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'delishs-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .delishs-el-button',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'delishs-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .delishs-el-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['delishs_image']['url']) ) {
            $delishs_image = !empty($settings['delishs_image']['id']) ? wp_get_attachment_image_url( $settings['delishs_image']['id'], 'full' ) : $settings['delishs_image']['url'];
            $delishs_image_alt = get_post_meta($settings["delishs_image"]["id"], "_wp_attachment_image_alt", true);
        }

        if ( !empty($settings['delishs_image_two']['url']) ) {
            $delishs_image_two = !empty($settings['delishs_image_two']['id']) ? wp_get_attachment_image_url( $settings['delishs_image_two']['id'], 'full' ) : $settings['delishs_image_two']['url'];
            $delishs_image_two_alt = get_post_meta($settings["delishs_image_two"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

    <?php if ( $settings['design_style']  == 'layout-1' ): ?>

        <section class="delishs-el-section about-our-company section-space overflow-hidden">
            <div class="container">
                <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                    <div class="about-our-company_shape">
                        <div class="about-our-company_shape-common about-our-company_shape-1 leftRight" data-background="<?php print get_template_directory_uri(); ?>/assets/imgs/about-our-company/about-our-company__shap-1.png"></div>
                        <div class="about-our-company_shape-common about-our-company_shape-2 rightLeft" data-background="<?php print get_template_directory_uri(); ?>/assets/imgs/about-our-company/about-our-company__shap-2.png"></div>
                    </div>
                <?php endif; ?>
                <div class="row g-24 align-items-center">
                    <div class="col-lg-6">
                        <div class="about-our-company__media wow clip-a-z">
                            <img class="img-fluid" src="<?php print esc_url($delishs_image); ?>" alt="<?php print esc_attr($delishs_image_alt); ?>">
                        </div>
                    </div>
                    <div class="col-lg-6">
                        <div class="section__title-wrapper about-our-company__content">
                            <?php if(!empty($settings['section_subheading'])): ?>
                                <span class="delishs-el-section-subheading section__subtitle justify-content-center mb-20 mb-xs-15 wow clip-a-z">
                                    <?php echo rrdevs_kses($settings['section_subheading']); ?>
                                </span>
                            <?php endif; ?>
                            <?php if ( !empty($settings['section_title' ]) ) : ?>
                                <h2 class="delishs-el-section-title section__title section__title-line-height-50 mb-35 mb-xs-30 title-animation">
                                    <?php print rrdevs_kses( $settings['section_title' ] ); ?>
                                </h2>
                            <?php endif; ?>
                            <?php if ( !empty($settings['section_description' ]) ) : ?>
                                <p class="delishs-el-section-description mb-40 mb-xs-30 section__p-18">
                                    <?php print rrdevs_kses( $settings['section_description' ] ); ?>
                                </p>
                            <?php endif; ?>

                            <div class="about-our-company__content-btn">
                                <?php if ( $settings['phone_icon_type']  == 'image' ): ?>
                                    <div class="about-our-company__content-btn-icon">
                                        <img class="img-fluid" src="<?php echo $settings['phone_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($settings['phone
                                        phone_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                    </div>
                                <?php elseif ( $settings['phone_icon_type']  == 'icon' ): ?>
                                    <div class="about-our-company__content-btn-icon">
                                        <?php delishs_render_icon($settings, 'phone_icon' ); ?>
                                    </div>
                                <?php endif; ?>
                                <div class="about-our-company__content-btn-text">
                                    <?php if ( !empty( $settings['phone_label'] ) ) : ?>
                                        <span><?php print rrdevs_kses($settings['phone_label']); ?></span>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['phone_number'] ) ) : ?>
                                        <h6>
                                            <a href="tel:<?php print esc_attr($settings['phone_number_link']); ?>">
                                                <?php print rrdevs_kses($settings['phone_number']); ?>
                                            </a>
                                        </h6>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>

    <?php elseif ( $settings['design_style']  == 'layout-2' ): 
        
        // Link
        if ('2' == $settings['delishs_button_link_type']) {
            $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['btn_page_link']));
            $this->add_render_attribute('delishs-button-arg', 'target', '_self');
            $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__red wow clip-a-z');
        } else {
            if ( ! empty( $settings['delishs_button_link']['url'] ) ) {
                $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn rr-btn__red wow clip-a-z');
            }
        }
        
        ?>

            <section class="delishs-el-section about-2 section-space overflow-hidden">
                <div class="container">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="about-2__shape">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/about-2/burger.png" class="img-fluid leftRight" alt="not found">
                        </div>
                    <?php endif; ?>
                    <div class="row g-24 align-items-center">
                        <div class="col-xl-6">
                            <div class="about-2__content">
                                <div class="section__title-wrapper text-start mb-35 mb-xs-40">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <span class="delishs-el-section-subheading section__subtitle section__subtitle-line-height justify-content-start mb-15 mb-xs-5 wow clip-a-z">
                                            <?php echo rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty($settings['section_title' ]) ) : ?>
                                        <h2 class="delishs-el-section-title section__title mb-0 title-animation">
                                            <?php print rrdevs_kses( $settings['section_title' ] ); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>

                                <div class="about-2__content-item__wrapper mb-40">
                                    <?php foreach ($settings['feature_list'] as $key => $item) : ?>
                                        <div class="about-2__content-item mb-35">
                                            <?php if ( $item['feature_icon_type']  == 'image' ): ?>
                                                <div class="about-2__content-item__media">
                                                    <img src="<?php echo $item['feature_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['feature_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                                </div>
                                            <?php elseif ( $item['feature_icon_type']  == 'icon' ): ?>
                                                <div class="about-2__content-item__media">
                                                    <?php delishs_render_icon($item, 'feature_icon' ); ?>
                                                </div>
                                            <?php endif; ?>
                                            <div class="about-2__content-item__text">
                                                <?php if ( !empty( $item['feature_text'] ) ) : ?>
                                                    <h6 class="text-uppercase"><?php print rrdevs_kses($item['feature_text']); ?></h6>
                                                <?php endif; ?>
                                                <?php if ( !empty( $item['feature_description'] ) ) : ?>
                                                    <p class="mb-0"><?php print rrdevs_kses($item['feature_description']); ?></p>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>

                                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                    <span class="hover-rl"></span>
                                    <span class="fake_hover"></span>
                                    <span class="btn-wrap">
                                    <span class="text-one">
                                        <?php print rrdevs_kses($settings['delishs_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                    </span>
                                    <span class="text-two">
                                        <?php print rrdevs_kses($settings['delishs_button_text']); ?> <img src="<?php print get_template_directory_uri(); ?>/assets/imgs/icon/arrow-right-2.svg" alt="not found">
                                    </span>
                                </span>
                                </a>
                            </div>
                        </div>

                        <div class="col-xl-6">
                            <div class="about-2__media wow clip-a-z">
                                <img src="<?php print esc_url($delishs_image); ?>" class="img-fluid" alt="<?php print esc_attr($delishs_image_alt); ?>">
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-3' ): 
            // Link
            if ('2' == $settings['delishs_button_link_type']) {
                $this->add_render_attribute('delishs-button-arg', 'href', get_permalink($settings['btn_page_link']));
                $this->add_render_attribute('delishs-button-arg', 'target', '_self');
                $this->add_render_attribute('delishs-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn-solid wow clip-a-z');
            } else {
                if ( ! empty( $settings['delishs_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'delishs-button-arg', $settings['delishs_button_link'] );
                    $this->add_render_attribute('delishs-button-arg', 'class', 'delishs-el-button rr-btn-solid wow clip-a-z');
                }
            }
        
        ?>

        <section class="delishs-el-section reservation section-space__bottom section-bg-3">
            <div class="container">
                <div class="row">
                    <div class="col-12">
                        <div class="reservation-wrapper">
                            <div class="reservation__content">
                                <div class="section__title-wrapper text-center text-xxl-start mb-45">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <span class="delishs-el-section-subheading section__subtitle justify-content-start mb-20 mb-xs-5 wow clip-a-z">
                                            <?php echo rrdevs_kses($settings['section_subheading']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if ( !empty($settings['section_title' ]) ) : ?>
                                        <h2 class="delishs-el-section-title section__title section__title-line-height-50 mb-30 title-animation">
                                            <?php print rrdevs_kses( $settings['section_title' ] ); ?>
                                        </h2>
                                    <?php endif; ?>
                                    <?php if ( !empty($settings['section_description' ]) ) : ?>
                                        <p class="delishs-el-section-description mb-0">
                                            <?php print rrdevs_kses( $settings['section_description' ] ); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>

                                <a <?php echo $this->get_render_attribute_string( 'delishs-button-arg' ); ?>>
                                    <span class="btn-wrap">
                                        <span class="text-one"><?php print rrdevs_kses($settings['delishs_button_text']); ?></span>
                                        <span class="text-two"><?php print rrdevs_kses($settings['delishs_button_text']); ?></span>
                                    </span>
                                </a>
                            </div>

                            <div class="reservation__media">
                                <img src="<?php print esc_url($delishs_image); ?>" class="image-1 wow clip-a-z" alt="<?php print esc_attr($delishs_image_alt); ?>">
                                <div class="reservation__media-small-thumb wow clip-a-z">
                                    <img class="image-2" src="<?php print esc_url($delishs_image_two); ?>" alt="<?php print esc_attr($delishs_image_two_alt); ?>">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>

        <?php endif;
	}

}

$widgets_manager->register( new Delishs_About() );